package cn.allbs.cache;

import cn.allbs.cache.listener.CacheMessageListener;
import cn.allbs.cache.properties.CacheConfigProperties;
import cn.allbs.cache.sequence.IdGeneratorUtil;
import cn.allbs.cache.support.RedisCaffeineCacheManager;
import cn.allbs.cache.support.RedisCaffeineCacheManagerCustomizer;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.data.redis.RedisAutoConfiguration;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.listener.ChannelTopic;
import org.springframework.data.redis.listener.RedisMessageListenerContainer;
import org.springframework.data.redis.serializer.GenericJackson2JsonRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;

import java.util.Objects;

/**
 * 功能:
 *
 * @author ChenQi
 * @version 1.0
 * @since 2021/3/20
 */
@Configuration(proxyBeanMethods = false)
@AutoConfigureAfter(RedisAutoConfiguration.class)
@EnableConfigurationProperties(CacheConfigProperties.class)
@Slf4j
public class MultilevelCacheAutoConfiguration {

    @Bean
    @ConditionalOnMissingBean(RedisSerializer.class)
    public RedisSerializer<Object> redisSerializer(ObjectProvider<ObjectMapper> objectProvider) {
        // jackson findAndRegisterModules，use copy
        ObjectMapper objectMapper = objectProvider.getIfAvailable(ObjectMapper::new).copy();
        // findAndRegisterModules
        objectMapper.findAndRegisterModules();
        // class type info to json
        GenericJackson2JsonRedisSerializer.registerNullValueSerializer(objectMapper, null);
        objectMapper.activateDefaultTyping(objectMapper.getPolymorphicTypeValidator(), ObjectMapper.DefaultTyping.NON_FINAL, JsonTypeInfo.As.PROPERTY);
        return new GenericJackson2JsonRedisSerializer(objectMapper);
    }

    @Bean(name = "stringKeyRedisTemplate")
    @ConditionalOnMissingBean(name = "stringKeyRedisTemplate")
    public RedisTemplate<Object, Object> stringKeyRedisTemplate(RedisConnectionFactory redisConnectionFactory, RedisSerializer<Object> redisSerializer) {
        RedisTemplate<Object, Object> template = new RedisTemplate<>();
        template.setConnectionFactory(redisConnectionFactory);
        template.setKeySerializer(new StringRedisSerializer());
        template.setHashKeySerializer(new StringRedisSerializer());
        // 使用Jackson2JsonRedisSerialize 替换默认序列化会明文储存
        template.setValueSerializer(redisSerializer);
        template.setHashValueSerializer(redisSerializer);
        template.setConnectionFactory(redisConnectionFactory);
        return template;
    }

    @Bean
    @ConditionalOnMissingBean
    public RedisCaffeineCacheManager cacheManager(CacheConfigProperties cacheConfigProperties, @Qualifier("stringKeyRedisTemplate") RedisTemplate<Object, Object> stringKeyRedisTemplate, ObjectProvider<RedisCaffeineCacheManagerCustomizer> cacheManagerCustomizers) {
        RedisCaffeineCacheManager cacheManager = new RedisCaffeineCacheManager(cacheConfigProperties, stringKeyRedisTemplate);
        cacheManagerCustomizers.orderedStream().forEach(customizer -> customizer.customize(cacheManager));
        return cacheManager;
    }

    @Bean
    @ConditionalOnMissingBean(name = "cacheMessageListenerContainer")
    public RedisMessageListenerContainer cacheMessageListenerContainer(CacheConfigProperties cacheConfigProperties, @Qualifier("stringKeyRedisTemplate") RedisTemplate<Object, Object> stringKeyRedisTemplate, @Qualifier("cacheMessageListener") CacheMessageListener cacheMessageListener) {
        RedisMessageListenerContainer redisMessageListenerContainer = new RedisMessageListenerContainer();
        redisMessageListenerContainer.setConnectionFactory(Objects.requireNonNull(stringKeyRedisTemplate.getConnectionFactory()));
        redisMessageListenerContainer.addMessageListener(cacheMessageListener, new ChannelTopic(cacheConfigProperties.getRedis().getTopic()));
        return redisMessageListenerContainer;
    }

    @Bean
    @SuppressWarnings("unchecked")
    @ConditionalOnMissingBean(name = "cacheMessageListener")
    public CacheMessageListener cacheMessageListener(@Qualifier("stringKeyRedisTemplate") RedisTemplate<Object, Object> stringKeyRedisTemplate, RedisCaffeineCacheManager redisCaffeineCacheManager) {
        return new CacheMessageListener((RedisSerializer<Object>) stringKeyRedisTemplate.getValueSerializer(), redisCaffeineCacheManager);
    }

    @Bean
    public IdGeneratorUtil idGeneratorUtil(@Qualifier("stringKeyRedisTemplate") RedisTemplate<Object, Object> stringKeyRedisTemplate) {
        return new IdGeneratorUtil(stringKeyRedisTemplate);
    }
}
