package cn.allbs.cache.sequence;

import cn.allbs.common.constant.StringPool;
import org.springframework.data.redis.core.RedisTemplate;

import java.time.*;
import java.time.format.DateTimeFormatter;

/**
 * @author ChenQi
 */
public class IdGeneratorUtil {

    private final SequenceManage sequenceManage;

    public IdGeneratorUtil(RedisTemplate<Object, Object> redisTemplate) {
        this.sequenceManage = new SequenceManage(redisTemplate);
    }

    /**
     * 生成id（每日重置自增序列）
     * 格式：日期 + 指定位位自增数
     * 如：20210804000001
     *
     * @param key    储存序号的redis key
     * @param length 生成的后缀长度
     * @return 日期 + 指定位位自增数序号
     */
    public String generateYMDId(String key, Integer length) {
        long num = sequenceManage.increment(key, getEndTime());
        return LocalDate.now().format(DateTimeFormatter.ofPattern("yyyyMMdd")) + String.format("%0" + length + "d", num);
    }

    /**
     * 生成指定开头的自增序列
     * 格式: 指定字符 + 指定位自增数
     * 如W00001
     *
     * @param key    redis 储存的key
     * @param prefix 指定的开头
     * @param length 格式化长度
     * @return 指定字符 + 指定位自增数
     */
    public String generatePreId(String key, String prefix, Integer length) {
        long num = sequenceManage.increment(key + StringPool.COLON + prefix);
        return prefix + String.format("%0" + length + "d", num);
    }

    /**
     * 生成指定开头的自增序列
     * 格式: 指定字符 + 指定位自增数
     * 如W00001
     *
     * @param key    redis 储存的key
     * @param prefix 指定的开头
     * @param length 格式化长度
     * @return 指定字符 + 指定位自增数
     */
    public String generatePreIdStep(String key, String prefix, Integer length, int step) {
        long num = sequenceManage.increment(key + StringPool.COLON + prefix, step);
        return prefix + String.format("%0" + length + "d", num);
    }

    /**
     * 生成指定开头的自增序列
     * 格式: 指定字符 + 指定位自增数
     * 如W00001
     *
     * @param key    redis 储存的key
     * @param prefix 指定的开头
     * @param length 格式化长度
     * @param step   所在步长
     * @return 指定字符 + 指定位自增数
     */
    public String generatePreId(String key, String prefix, Integer length, int step) {
        long num = sequenceManage.increment(key + StringPool.COLON + prefix, step);
        return prefix + String.format("%0" + length + "d", num);
    }

    /**
     * 初始化发号器默认值
     *
     * @param key          redis 的key
     * @param prefix       序列化的编码起始字符串
     * @param initialValue 初始值
     */
    public void initGenerate(String key, String prefix, int initialValue) {
        sequenceManage.initIncrement(key + StringPool.COLON + prefix, initialValue);
    }

    /**
     * 初始化发号器默认值
     *
     * @param key          redis 的key
     * @param initialValue 初始值
     */
    public void initGenerate(String key, int initialValue) {
        sequenceManage.initIncrement(key, initialValue);
    }

    /**
     * 获取当天的结束时间
     */
    public Instant getEndTime() {
        LocalDateTime endTime = LocalDateTime.of(LocalDate.now(), LocalTime.MAX);
        return endTime.toInstant(ZoneOffset.ofHours(8));
    }
}
