/**
 * Copyright (c) 2023 James Zhan 詹波 (zhanbocn@126.com)
 * Aifei Enjoy is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cn.aifei.kit;

import java.security.MessageDigest;
import java.util.concurrent.ThreadLocalRandom;

public class HashKit {

	public static final long FNV_OFFSET_BASIS_64 = 0xcbf29ce484222325L;
	public static final long FNV_PRIME_64 = 0x100000001b3L;

	// private static final java.security.SecureRandom random = new java.security.SecureRandom();
	private static final char[] HEX_DIGITS = "0123456789abcdef".toCharArray();
	private static final char[] CHAR_ARRAY = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ".toCharArray();

	public static long fnv1a64(String key) {
		long hash = FNV_OFFSET_BASIS_64;
		for(int i=0, size=key.length(); i<size; i++) {
			hash ^= key.charAt(i);
			hash *= FNV_PRIME_64;
		}
		return hash;
	}

	public static String md5(String srcStr){
		return hash("MD5", srcStr);
	}

	public static String sha1(String srcStr){
		return hash("SHA-1", srcStr);
	}

	public static String sha256(String srcStr){
		return hash("SHA-256", srcStr);
	}

	public static String sha384(String srcStr){
		return hash("SHA-384", srcStr);
	}

	public static String sha512(String srcStr){
		return hash("SHA-512", srcStr);
	}

	public static String hash(String algorithm, String srcStr) {
		try {
			MessageDigest md = MessageDigest.getInstance(algorithm);
			byte[] bytes = md.digest(srcStr.getBytes("utf-8"));
			return toHex(bytes);
		}
		catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	public static String toHex(byte[] bytes) {
		StringBuilder ret = new StringBuilder(bytes.length * 2);
		for (int i=0; i<bytes.length; i++) {
			ret.append(HEX_DIGITS[(bytes[i] >> 4) & 0x0f]);
			ret.append(HEX_DIGITS[bytes[i] & 0x0f]);
		}
		return ret.toString();
	}

	/**
	 * md5 128bit 16bytes
	 * sha1 160bit 20bytes
	 * sha256 256bit 32bytes
	 * sha384 384bit 48bytes
	 * sha512 512bit 64bytes
	 */
	public static String generateSalt(int saltLength) {
		StringBuilder salt = new StringBuilder(saltLength);
		ThreadLocalRandom random = ThreadLocalRandom.current();
		for (int i=0; i<saltLength; i++) {
			salt.append(CHAR_ARRAY[random.nextInt(CHAR_ARRAY.length)]);
		}
		return salt.toString();
	}

	public static String generateSaltForSha256() {
		return generateSalt(32);
	}

	public static String generateSaltForSha512() {
		return generateSalt(64);
	}

	// 生成随机字符串
	public static String generateRandomStr(int strLength) {
		return generateSalt(strLength);
	}

	public static boolean slowEquals(byte[] a, byte[] b) {
		if (a == null || b == null) {
			return false;
		}

		int diff = a.length ^ b.length;
		for(int i=0; i<a.length && i<b.length; i++) {
			diff |= a[i] ^ b[i];
		}
		return diff == 0;
    }
}




