/**
 * Copyright (c) 2023 James Zhan 詹波 (zhanbocn@126.com)
 * Aifei Enjoy is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cn.aifei.enjoy.stat.ast;

import java.io.IOException;
import java.nio.charset.Charset;
import cn.aifei.enjoy.EngineConfig;
import cn.aifei.enjoy.Env;
import cn.aifei.enjoy.TemplateException;
import cn.aifei.enjoy.io.IWritable;
import cn.aifei.enjoy.io.Writer;
import cn.aifei.enjoy.stat.Compressor;
import cn.aifei.enjoy.stat.Scope;

/**
 * Text 输出纯文本块以及使用 "#[[" 与 "]]#" 定义的原样输出块
 */
public class Text extends Stat implements IWritable {

	// content、bytes、chars 三者必有一者不为 null
	// 在 OutputStream、Writer 混合模式下 bytes、chars 同时不为null
	private StringBuilder content;
	private Charset charset;
	private byte[] bytes;
	private char[] chars;

	// content 初始值在 Lexer 中已确保不为 null
	public Text(StringBuilder content, EngineConfig ec) {
		Compressor c = ec.getCompressor();
		this.content = (c != null ? c.compress(content) : content);
		this.charset = Charset.forName(ec.getEncoding());
		this.bytes = null;
		this.chars = null;
	}

	public void exec(Env env, Scope scope, Writer writer) {
		try {
			writer.write(this);
		} catch (IOException e) {
			throw new TemplateException(e.getMessage(), location, e);
		}
	}

	public byte[] getBytes() {
		if (bytes != null) {
			return bytes;
		}

		synchronized (this) {
			if (bytes != null) {
				return bytes;
			}

			if (content != null) {
				bytes = content.toString().getBytes(charset);
				content = null;
				return bytes;
			} else {
				bytes = new String(chars).getBytes(charset);
				return bytes;
			}
		}
	}

	public char[] getChars() {
		if (chars != null) {
			return chars;
		}

		synchronized (this) {
			if (chars != null) {
				return chars;
			}

			if (content != null) {
				char[] charsTemp = new char[content.length()];
				content.getChars(0, content.length(), charsTemp, 0);
				chars = charsTemp;
				content = null;
				return chars;
			} else {
				String strTemp = new String(bytes, charset);
				char[] charsTemp = new char[strTemp.length()];
				strTemp.getChars(0, strTemp.length(), charsTemp, 0);
				chars = charsTemp;
				return chars;
			}
		}
	}

	public boolean isEmpty() {
		if (content != null) {
			return content.length() == 0;
		} else if (bytes != null) {
			return bytes.length == 0;
		} else {
			return chars.length == 0;
		}
	}

//	public String getContent() {
//		return text != null ? new String(text) : null;
//	}

	public String toString() {
		if (bytes != null) {
			return new String(bytes, charset);
		} else if (chars != null) {
			return new String(chars);
		} else {
			return content.toString();
		}
	}
}



