/**
 * Copyright (c) 2023 James Zhan 詹波 (zhanbocn@126.com)
 * Aifei Enjoy is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cn.aifei.enjoy.stat.ast;

import java.util.List;
import cn.aifei.enjoy.Env;
import cn.aifei.enjoy.TemplateException;
import cn.aifei.enjoy.io.Writer;
import cn.aifei.enjoy.stat.Ctrl;
import cn.aifei.enjoy.stat.Scope;

/**
 * StatList
 */
public class StatList extends Stat {

	public static final Stat NULL_STAT = NullStat.me;
	public static final Stat[] NULL_STAT_ARRAY = new Stat[0];

	private Stat[] statArray;

	public StatList(List<Stat> statList) {
		if (statList.size() > 0) {
			this.statArray = statList.toArray(new Stat[statList.size()]);
		} else {
			this.statArray = NULL_STAT_ARRAY;
		}
	}

	/**
	 * 持有 StatList 的指令可以通过此方法提升 AST 执行性能
	 * 1：当 statArray.length >  1 时返回 StatList 自身
	 * 2：当 statArray.length == 1 时返回 statArray[0]
	 * 3：其它情况返回 NullStat
	 *
	 * 意义在于，当满足前面两个条件时，避免掉了 StatList.exec(...) 方法中的判断与循环
	 */
	public Stat getActualStat() {
		if (statArray.length > 1) {
			return this;
		} else if (statArray.length == 1) {
			return statArray[0];
		} else {
			return NULL_STAT;
		}
	}

	public void exec(Env env, Scope scope, Writer writer) {
		Ctrl ctrl = scope.getCtrl();
		for (int i=0; i<statArray.length; i++) {
			if (ctrl.isJump()) {
				break ;
			}
			statArray[i].exec(env, scope, writer);
		}
	}

	public int length() {
		return statArray.length;
	}

	public Stat getStat(int index) {
		if (index < 0 || index >= statArray.length) {
			throw new TemplateException("Index out of bounds: index = " + index + ", length = " + statArray.length, location);
		}
		return statArray[index];
	}
}


