/**
 * Copyright (c) 2023 James Zhan 詹波 (zhanbocn@126.com)
 * Aifei Enjoy is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cn.aifei.enjoy.stat;

import cn.aifei.enjoy.expr.ast.ExprList;
import cn.aifei.enjoy.stat.ast.Output;

/**
 * OutputDirectiveFactory
 * 用于定制自定义输出指令，替换系统默认输出指令，满足个性化需求
 *
 * 用法：
 * 1：定义 MyOutput
 * public class MyOutput extends Output {
 *   public MyOutput(ExprList exprList) {
 *     super(exprList);
 *   }
 *
 *   public void exec(Env env, Scope scope, Writer writer) {
 *     Object value = exprList.eval(scope);
 *     if (value != null) {
 *     	  write(writer, value.toString());
 *     }
 *   }
 * }
 *
 * 2：定义 MyOutputDirectiveFactory
 * public class MyOutputDirectiveFactory extends OutputDirectiveFactory {
 *   public Output getOutputDirective(ExprList exprList) {
 *     return new MyOutput(exprList);
 *   }
 * }
 *
 * 3：配置
 * engine.setOutputDirectiveFactory(new MyOutputDirectiveFactory())
 */
public class OutputDirectiveFactory {

	public static final OutputDirectiveFactory me = new OutputDirectiveFactory();

	public Output getOutputDirective(ExprList exprList, Location location) {
		return new Output(exprList, location);
	}
}


