/**
 * Copyright (c) 2023 James Zhan 詹波 (zhanbocn@126.com)
 * Aifei Enjoy is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cn.aifei.enjoy.stat;

/**
 * CharTable 空间换时间优化字符判断性能
 * 负值参数强转 char 会自动变正值，无需判断负值数组下标
 * isLetter(EOF) 不会下标越界
 */
public class CharTable {

	private static final char NULL = 0;
	private static final char EN_SIZE = 128;
	private static final char CH_SIZE = 0x9FA5 + 1;

	private static char size = EN_SIZE;
	private static char[] letterChars = buildLetterChars();
	private static char[] letterOrDigitChars = buildLetterOrDigitChars();
	private static char[] exprChars = buildExprChars();

	private CharTable() {}

	/**
	 * 设置为 true 支持表达式、变量名、方法名、模板函数名使用中文
	 */
	public static void setChineseExpression(boolean enable) {
		if (enable) {
			size = CH_SIZE;
		} else {
			size = EN_SIZE;
		}

		letterChars = buildLetterChars();
		letterOrDigitChars = buildLetterOrDigitChars();
		exprChars = buildExprChars();
	}

	// 添加中文字符，Unicode 编码范围：4E00-9FA5
	private static void addChineseChar(char[] ret) {
		if (ret.length == CH_SIZE) {
			for (char i=0x4E00; i<CH_SIZE; i++) {
				ret[i] = i;
			}
		}
	}

	private static char[] createCharArray() {
		char[] ret = new char[size];
		for (char i=0; i<size; i++) {
			ret[i] = NULL;
		}

		addChineseChar(ret);

		return ret;
	}

	private static char[] buildLetterChars() {
		char[] ret = createCharArray();
		for (char i='a'; i<='z'; i++) {
			ret[i] = i;
		}
		for (char i='A'; i<='Z'; i++) {
			ret[i] = i;
		}
		ret['_'] = '_';			// 包含下划线字符 '_'
		ret['$'] = '$';			// 变量名、方法名支持字符 '$'
		return ret;
	}

	private static char[] buildLetterOrDigitChars() {
		char[] ret = buildLetterChars();
		for (char i='0'; i<='9'; i++) {
			ret[i] = i;
		}
		return ret;
	}

	private static char[] buildExprChars() {
		char[] ret = createCharArray();
		ret['\t'] = '\t';
		ret['\n'] = '\n';
		ret['\r'] = '\r';
		for (char i=' '; i<='}'; i++) {
			ret[i] = i;
		}

		ret['#'] = NULL;
		// ret['$'] = NULL;    // 变量名、方法名支持字符 '$'
		ret['@'] = NULL;
		ret['\\'] = NULL;
		ret['^'] = NULL;
		ret['`'] = NULL;
		return ret;
	}

	public static boolean isLetter(char c) {
		return c < size && letterChars[c] != NULL;
	}

	public static boolean isLetterOrDigit(char c) {
		return c < size && letterOrDigitChars[c] != NULL;
	}

	public static boolean isExprChar(char c) {
		return c < size && exprChars[c] != NULL;
	}

	public static boolean isDigit(char c) {
		return c >= '0' && c <= '9';
	}

	public static boolean isBlank(char c) {
		return c == ' ' || c == '\t' || c == '\r';								// \t\r\u000C
	}

	public static boolean isBlankOrLineFeed(char c) {
		return c == ' ' || c == '\t' || c == '\r' || c == '\n';		// \t\r\n\u000C
	}

	public static boolean isHexadecimalDigit(char c) {
		return (c >= '0' && c <= '9') || (c >= 'A' && c <= 'F') || (c >= 'a' && c <= 'f');
	}

	public static boolean isOctalDigit(char c) {
		return c >= '0' && c <= '7';
	}
}




