/**
 * Copyright (c) 2023 James Zhan 詹波 (zhanbocn@126.com)
 * Aifei Enjoy is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cn.aifei.enjoy.source;

import cn.aifei.enjoy.EngineConfig;
import cn.aifei.kit.HashKit;
import cn.aifei.kit.StrKit;

/**
 * StringSource 用于从 String 变量中加载模板内容
 */
public class StringSource implements ISource {

	private String cacheKey;
	private StringBuilder content;

	/**
	 * 构造 StringSource
	 * @param content 模板内容
	 * @param cache true 则缓存 Template，否则不缓存
	 */
	public StringSource(String content, boolean cache) {
		this(content, cache ? HashKit.md5(content) : null);
	}

	/**
	 * 构造 StringSource
	 * @param content 模板内容
	 * @param cacheKey 缓存 Template 使用的 key，值为 null 时不缓存
	 */
	public StringSource(String content, String cacheKey) {
		if (StrKit.isBlank(content)) {
			throw new IllegalArgumentException("content can not be blank");
		}
		this.content = new StringBuilder(content);
		this.cacheKey = cacheKey;
	}

	public StringSource(StringBuilder content, boolean cache) {
		this(content, cache && content != null ? HashKit.md5(content.toString()) : null);
	}

	public StringSource(StringBuilder content, String cacheKey) {
		if (content == null || content.length() == 0) {
			throw new IllegalArgumentException("content can not be blank");
		}
		this.content = content;
		this.cacheKey = cacheKey;			// cacheKey 值为 null 时不缓存
	}

	public boolean isModified() {
		return false;
	}

	public String getCacheKey() {
		return cacheKey;
	}

	public StringBuilder getContent() {
		return content;
	}

	public String getEncoding() {
		return EngineConfig.DEFAULT_ENCODING;
	}

	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("cacheKey : ").append(cacheKey).append("\n");
		sb.append("content : ").append(content).append("\n");
		return sb.toString();
	}
}







