/**
 * Copyright (c) 2023 James Zhan 詹波 (zhanbocn@126.com)
 * Aifei Enjoy is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cn.aifei.enjoy.source;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;

import cn.aifei.enjoy.EngineConfig;

/**
 * FileSource 用于从普通文件中加载模板内容
 */
public class FileSource implements ISource {

	private String finalFileName;
	private String fileName;
	private String encoding;

	private long lastModified;

	public FileSource(String baseTemplatePath, String fileName, String encoding) {
		this.finalFileName = buildFinalFileName(baseTemplatePath, fileName);
		this.fileName = fileName;
		this.encoding= encoding;
	}

	public FileSource(String baseTemplatePath, String fileName) {
		this(baseTemplatePath, fileName, EngineConfig.DEFAULT_ENCODING);
	}

	public boolean isModified() {
		return lastModified != new File(finalFileName).lastModified();
	}

	public String getCacheKey() {
		return fileName;
	}

	public String getEncoding() {
		return encoding;
	}

	public String getFinalFileName() {
		return finalFileName;
	}

	public String getFileName() {
		return fileName;
	}

	public StringBuilder getContent() {
		File file = new File(finalFileName);
		if (!file.exists()) {
			throw new RuntimeException("File not found : \"" + finalFileName + "\"");
		}

		// 极为重要，否则在开发模式下 isModified() 一直返回 true，缓存一直失效（原因是 lastModified 默认值为 0）
		this.lastModified = file.lastModified();

		return loadFile(file, encoding);
	}

	private String buildFinalFileName(String baseTemplatePath, String fileName) {
		if (baseTemplatePath == null) {
			return fileName;
		}
		char firstChar = fileName.charAt(0);
		String finalFileName;
		if (firstChar == '/' || firstChar == '\\') {
			finalFileName = baseTemplatePath + fileName;
		} else {
			finalFileName = baseTemplatePath + File.separator + fileName;
		}
		return finalFileName;
	}

	public static StringBuilder loadFile(File file, String encoding) {
		try (InputStreamReader isr = new InputStreamReader(new FileInputStream(file), encoding)) {
			StringBuilder ret = new StringBuilder((int)file.length() + 3);
			char[] buf = new char[1024];
			for (int num; (num = isr.read(buf, 0, buf.length)) != -1;) {
				ret.append(buf, 0, num);
			}
			return ret;

		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("File name: ").append(fileName).append("\n");
		sb.append("Final file name: ").append(finalFileName).append("\n");
		sb.append("Last modified: ").append(lastModified).append("\n");
		return sb.toString();
	}
}




