/**
 * Copyright (c) 2023 James Zhan 詹波 (zhanbocn@126.com)
 * Aifei Enjoy is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cn.aifei.enjoy.ext.sharedmethod;

import java.lang.reflect.Array;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

/**
 * SharedMethodLib 共享方法库，逐步添加一些最常用的共享方法
 *
 * <br>
 * 3.3 版本之前的 Logic.isTrue(Object) 方法不再对  Collection、
 * Map、数组、Iterator、Iterable 进行为空的判断，这部分逻辑已转移至
 * SharedMethodLib.isEmpty(Object)
 */
public class SharedMethodLib {

	/**
	 * 判断 Collection、Map、数组、Iterator、Iterable 类型对象中的元素个数是否为 0
	 * 规则：
	 * 1：null 返回 true
	 * 2：List、Set 等一切继承自 Collection 的，返回 isEmpty()
	 * 3：Map 返回 isEmpty()
	 * 4：数组返回 length == 0
	 * 5：Iterator 返回  ! hasNext()
	 * 6：Iterable 返回  ! iterator().hasNext()
	 *
	 * 注意：原先 Logic.isTrue(Object) 中对集合与数组类型为空的判断转移到此方法中
	 */
	public Boolean isEmpty(Object v) {
		if (v == null) {
			return true;
		}

		if (v instanceof Collection) {
			return ((Collection<?>)v).isEmpty();
		}
		if (v instanceof Map) {
			return ((Map<?, ?>)v).isEmpty();
		}

		if (v.getClass().isArray()) {
			return Array.getLength(v) == 0;
		}

		if (v instanceof Iterator) {
			return ! ((Iterator<?>)v).hasNext();
		}
		if (v instanceof Iterable) {
			return ! ((Iterable<?>)v).iterator().hasNext();
		}

		throw new IllegalArgumentException("isEmpty(...) 方法只能接受 Collection、Map、数组、Iterator、Iterable 类型参数");
	}

	public Boolean notEmpty(Object v) {
		return !isEmpty(v);
	}
}






