/**
 * Copyright (c) 2023 James Zhan 詹波 (zhanbocn@126.com)
 * Aifei Enjoy is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cn.aifei.enjoy.ext.directive;

import cn.aifei.enjoy.Directive;
import cn.aifei.enjoy.Env;
import cn.aifei.enjoy.expr.ast.Const;
import cn.aifei.enjoy.expr.ast.Expr;
import cn.aifei.enjoy.expr.ast.ExprList;
import cn.aifei.enjoy.expr.ast.Id;
import cn.aifei.enjoy.io.CharWriter;
import cn.aifei.enjoy.io.FastStringWriter;
import cn.aifei.enjoy.io.Writer;
import cn.aifei.enjoy.stat.ParseException;
import cn.aifei.enjoy.stat.Scope;

/**
 * #string 指令方便定义大量的多行文本变量，这个是 java 语言中极为需要的功能
 *
 * 定义：
 * #string(name)
 *    在此是大量的字符串
 * #end
 *
 * 使用：
 * #(name)
 */
public class StringDirective extends Directive {

	private String name;
	private boolean isLocalAssignment = false;

	public void setExprList(ExprList exprList) {
		Expr[] exprArray = exprList.getExprArray();
		if (exprArray.length == 0) {
			throw new ParseException("#string directive parameter cant not be null", location);
		}
		if (exprArray.length > 2) {
			throw new ParseException("wrong number of #string directive parameter, two parameters allowed at most", location);
		}

		if (!(exprArray[0] instanceof Id)) {
			throw new ParseException("#string first parameter must be identifier", location);
		}
		this.name = ((Id)exprArray[0]).getId();
		if (exprArray.length == 2) {
			if (exprArray[1] instanceof Const) {
				if (((Const)exprArray[1]).isBoolean()) {
					this.isLocalAssignment = ((Const)exprArray[1]).getBoolean();
				} else {
					throw new ParseException("#string sencond parameter must be boolean", location);
				}
			}
		}
	}

	public void exec(Env env, Scope scope, Writer writer) {
		CharWriter charWriter = new CharWriter(64);
		FastStringWriter fsw = new FastStringWriter();
		charWriter.init(fsw);
		try {
			stat.exec(env, scope, charWriter);
		} finally {
			charWriter.close();
		}

		if (this.isLocalAssignment) {
			scope.setLocal(name, fsw.toString());
		} else {
			scope.set(name, fsw.toString());
		}
	}

	/**
	 * hasEnd() 方法返回 true 时，表示该指令拥有指令体以及 #end 结束块
	 * 模板引擎在解析时会将 "指令体" 赋值到 stat 属性中，在 exec(...) 方法中
	 * 可通过 stat.exec(...) 执行 "指令体" 内部的所有指令
	 */
	public boolean hasEnd() {
		return true;
	}
}








