/**
 * Copyright (c) 2023 James Zhan 詹波 (zhanbocn@126.com)
 * Aifei Enjoy is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cn.aifei.enjoy.ext.directive;

import java.math.RoundingMode;
import java.text.DecimalFormat;

import cn.aifei.enjoy.Directive;
import cn.aifei.enjoy.Env;
import cn.aifei.enjoy.TemplateException;
import cn.aifei.enjoy.expr.ast.Expr;
import cn.aifei.enjoy.expr.ast.ExprList;
import cn.aifei.enjoy.io.Writer;
import cn.aifei.enjoy.stat.ParseException;
import cn.aifei.enjoy.stat.Scope;

/**
 * #number 数字格式化输出指令
 *
 * 优化时要注意 DecimalFormat 并非线程安全
 *
 * 两种用法：
 * 1：#number(n) 用默认 pattern 输出变量中的值
 * 2：#number(n, "#.##") 用第二个参数指定的 pattern 输出变量中的值
 *
 * 注意：
 * 1：pattern 的使用与 java.text.DecimalFormat 的完全一样
 *    在拿不定主意的时候可以在搜索引擎中搜索关键字：DecimalFormat
 * 2：#number 指令中的参数可以是变量，例如：#number(n, p) 中的 n 与 p 可以全都是变量
 *
 * <pre>
 * 示例：
 * #number(3.1415926, "#.##")
 * #number(0.9518, "#.##%")
 * #number(300000, "光速为每秒 ,### 公里。")
 * #number(1299792458, ",###")	// 每三位以逗号进行分隔输出为：1,299,792,458
 *
 * #set(n = 1.234)
 * #set(p = "#.##")
 * #number(n, p)
 * </pre>
 */
public class NumberDirective extends Directive {

	private Expr valueExpr;
	private Expr patternExpr;

	public void setExprList(ExprList exprList) {
		int paraNum = exprList.length();
		if (paraNum == 0) {
			throw new ParseException("The parameter of #number directive can not be blank", location);
		}
		if (paraNum > 2) {
			throw new ParseException("Wrong number parameter of #number directive, two parameters allowed at most", location);
		}

		valueExpr = exprList.getExpr(0);
		patternExpr = (paraNum == 1 ? null : exprList.getExpr(1));
	}

	public void exec(Env env, Scope scope, Writer writer) {
		Object value = valueExpr.eval(scope);
		if (value == null) {
			return ;
		}

		RoundingMode roundingMode = env.getEngineConfig().getRoundingMode();
		if (patternExpr == null) {
			outputWithoutPattern(value, roundingMode, writer);
		} else {
			outputWithPattern(value, roundingMode, scope, writer);
		}
	}

	private void outputWithoutPattern(Object value, RoundingMode roundingMode, Writer writer) {
		DecimalFormat df = new DecimalFormat();
		df.setRoundingMode(roundingMode);

		String ret = df.format(value);
		write(writer, ret);
	}

	private void outputWithPattern(Object value, RoundingMode roundingMode, Scope scope, Writer writer) {
		Object pattern = patternExpr.eval(scope);
		if ( !(pattern instanceof String) ) {
			throw new TemplateException("The sencond parameter pattern of #number directive must be String", location);
		}

		DecimalFormat df = new DecimalFormat((String)pattern);
		df.setRoundingMode(roundingMode);

		String ret = df.format(value);
		write(writer, ret);
	}
}





