/**
 * Copyright (c) 2023 James Zhan 詹波 (zhanbocn@126.com)
 * Aifei Enjoy is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cn.aifei.enjoy.ext.directive;

import java.util.Date;

import cn.aifei.enjoy.Directive;
import cn.aifei.enjoy.Env;
import cn.aifei.enjoy.TemplateException;
import cn.aifei.enjoy.expr.ast.ExprList;
import cn.aifei.enjoy.io.Writer;
import cn.aifei.enjoy.stat.ParseException;
import cn.aifei.enjoy.stat.Scope;

/**
 * 根据 datePattern 参数输出当前时间，未指定 datePattern 参数时默认使用以下配置
 * env.getEngineConfig().getDatePattern()
 *
 * 注意该指令需要配置才能使用：
 * engine.addDirective("now", NowDirective.class, true);
 *
 * 例子：
 * 1：#now()
 * 2：#now("HH:mm:ss")
 * 3：#now("yyyy-MM-dd HH:mm:ss")
 */
public class NowDirective extends Directive {

	public void setExprList(ExprList exprList) {
		if (exprList.length() > 1) {
			throw new ParseException("#now directive support one parameter only", location);
		}
		super.setExprList(exprList);
	}

	public void exec(Env env, Scope scope, Writer writer) {
		String datePattern;
		if (exprList.length() == 0) {
			datePattern = env.getEngineConfig().getDatePattern();
		} else {
			Object dp = exprList.eval(scope);
			if (dp instanceof String) {
				datePattern = (String)dp;
			} else {
				throw new TemplateException("The parameter of #now directive must be String", location);
			}
		}

		try {
			writer.write(new Date(), datePattern);
		} catch (Exception e) {
			throw new TemplateException(e.getMessage(), location, e);
		}
	}
}



