/**
 * Copyright (c) 2023 James Zhan 詹波 (zhanbocn@126.com)
 * Aifei Enjoy is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cn.aifei.enjoy.ext.directive;

import java.util.ArrayList;

import cn.aifei.enjoy.Directive;
import cn.aifei.enjoy.Env;
import cn.aifei.enjoy.TemplateException;
import cn.aifei.enjoy.expr.ast.Const;
import cn.aifei.enjoy.expr.ast.Expr;
import cn.aifei.enjoy.expr.ast.ExprList;
import cn.aifei.enjoy.stat.ast.Define;
import cn.aifei.enjoy.io.Writer;
import cn.aifei.enjoy.stat.ParseException;
import cn.aifei.enjoy.stat.Scope;

/**
 * CallDirective 动态调用模板函数
 *
 * 模板函数的名称与参数都可以动态指定，提升模板函数调用的灵活性
 *
 * 例如：
 *     #call(funcName, p1, p2, ..., pn)
 *     其中 funcName，为函数名，p1、p2、pn 为被调用函数所使用的参数
 *
 *
 * 如果希望模板函数不存在时忽略其调用，添加常量值 true 在第一个参数位置即可
 * 例如：
 *     #call(true, funcName, p1, p2, ..., pn)
 *
 *
 * TODO 后续优化看一下 ast.Call.java
 */
public class CallDirective extends Directive {

	protected Expr funcNameExpr;
	protected ExprList paraExpr;

	protected boolean nullSafe = false;		// 是否支持函数名不存在时跳过

	public void setExprList(ExprList exprList) {
		int len = exprList.length();
		if (len == 0) {
			throw new ParseException("Template function name required", location);
		}

		int index = 0;
		Expr expr = exprList.getExpr(index);
		if (expr instanceof Const && ((Const)expr).isBoolean()) {
			if (len == 1) {
				throw new ParseException("Template function name required", location);
			}

			nullSafe = ((Const)expr).getBoolean();
			index++;
		}

		funcNameExpr = exprList.getExpr(index++);

		ArrayList<Expr> list = new ArrayList<Expr>();
		for (int i=index; i<len; i++) {
			list.add(exprList.getExpr(i));
		}
		paraExpr = new ExprList(list);
	}

	public void exec(Env env, Scope scope, Writer writer) {
		Object funcNameValue = funcNameExpr.eval(scope);
		if (funcNameValue == null) {
			if (nullSafe) {
				return ;
			}
			throw new TemplateException("Template function name can not be null", location);
		}

		if (!(funcNameValue instanceof String)) {
			throw new TemplateException("Template function name must be String", location);
		}

		Define func = env.getFunction(funcNameValue.toString());

		if (func == null) {
			if (nullSafe) {
				return ;
			}
			throw new TemplateException("Template function not found : " + funcNameValue, location);
		}

		func.call(env, scope, paraExpr, writer);
	}
}




