/**
 * Copyright (c) 2023 James Zhan 詹波 (zhanbocn@126.com)
 * Aifei Enjoy is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cn.aifei.enjoy.expr.ast;

import cn.aifei.enjoy.TemplateException;
import cn.aifei.enjoy.stat.Location;
import cn.aifei.enjoy.stat.ParseException;
import cn.aifei.enjoy.stat.Scope;

/**
 * SharedMethod
 *
 * 用法：
 * engine.addSharedMethod(new StrKit());
 * engine.addSharedStaticMethod(MyKit.class);
 *
 * #if (notBlank(para))
 *     ....
 * #end
 *
 * 上面代码中的 notBlank 方法来自 StrKit
 */
public class SharedMethod extends Expr {

	private SharedMethodKit sharedMethodKit;
	private String methodName;
	private ExprList exprList;

	public SharedMethod(SharedMethodKit sharedMethodKit, String methodName, ExprList exprList, Location location) {
		if (MethodKit.isForbiddenMethod(methodName)) {
			throw new ParseException("Forbidden method: " + methodName, location);
		}
		this.sharedMethodKit = sharedMethodKit;
		this.methodName = methodName;
		this.exprList = exprList;
		this.location = location;
	}

	public Object eval(Scope scope) {
		Object[] argValues = exprList.evalExprList(scope);

		try {
			SharedMethodKit.SharedMethodInfo sharedMethodInfo = sharedMethodKit.getSharedMethodInfo(methodName, argValues);
			if (sharedMethodInfo != null) {
				return sharedMethodInfo.invoke(argValues);
			} else {
				// ShareMethod 相当于是固定的静态的方法，不支持 null safe，null safe 只支持具有动态特征的用法
				throw new TemplateException(Method.buildMethodNotFoundSignature("Shared method not found: ", methodName, argValues), location);
			}

		} catch (TemplateException | ParseException e) {
			throw e;
		} catch (Exception e) {
			throw new TemplateException(e.getMessage(), location, e);
		}
	}
}




