/**
 * Copyright (c) 2023 James Zhan 詹波 (zhanbocn@126.com)
 * Aifei Enjoy is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cn.aifei.enjoy.expr.ast;

import java.util.AbstractList;

import cn.aifei.enjoy.TemplateException;
import cn.aifei.enjoy.stat.Location;
import cn.aifei.enjoy.stat.ParseException;
import cn.aifei.enjoy.stat.Scope;

/**
 * RangeArray : [expr .. expr]
 *
 * 用法：
 * 1：[1..3]
 * 2：[3..1]
 */
public class RangeArray extends Expr {

	private Expr start;
	private Expr end;

	/**
	 * array : '[' exprList ? | range ? ']'
	 * exprList : expr (',' expr)*
	 * range : expr .. expr
	 */
	public RangeArray(Expr start, Expr end, Location location) {
		if (start == null) {
			throw new ParseException("The start value of range array can not be blank", location);
		}
		if (end == null) {
			throw new ParseException("The end value of range array can not be blank", location);
		}
		this.start = start;
		this.end = end;
		this.location = location;
	}

	public Object eval(Scope scope) {
		Object startValue = start.eval(scope);
		if ( !(startValue instanceof Integer) ) {
			throw new TemplateException("The start value of range array must be Integer", location);
		}
		Object endValue = end.eval(scope);
		if ( !(endValue instanceof Integer) ) {
			throw new TemplateException("The end value of range array must be Integer", location);
		}

		return new RangeList((Integer)startValue, (Integer)endValue, location);
	}

	public static class RangeList extends AbstractList<Integer> {

		final int start;
		final int size;
		final boolean increase;
		final Location location;

		public RangeList(int start, int end, Location location) {
			this.start = start;
			this.increase = (start <= end);
			this.size = Math.abs(end - start) + 1;
			this.location = location;
		}

		public Integer get(int index) {
			if (index < 0 || index >= size) {
				throw new TemplateException("Index out of bounds. Index: " + index + ", Size: " + size, location);
			}
			return increase ? start + index : start - index;
		}

		public int size() {
			return size;
		}
	}
}




