/**
 * Copyright (c) 2023 James Zhan 詹波 (zhanbocn@126.com)
 * Aifei Enjoy is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cn.aifei.enjoy.expr.ast;

import cn.aifei.enjoy.stat.Ctrl;
import cn.aifei.enjoy.stat.Location;
import cn.aifei.enjoy.stat.ParseException;
import cn.aifei.enjoy.stat.Scope;

/**
 * NullSafe
 * 在原则上只支持具有动态特征的用法，例如：方法调用、字段取值、Map 与 List 取值
 * 而不支持具有静态特征的用法，例如：static method 调用、shared method 调用
 *
 * 用法：
 * #( seoTitle ?? "Aifei 极速开发社区" )
 * 支持级联：  #( a.b.c ?? "Aifei 极速开发社区" )
 * 支持嵌套：  #( a ?? b ?? c ?? d)
 */
public class NullSafe extends Expr {

	private Expr left;
	private Expr right;

	public NullSafe(Expr left, Expr right, Location location) {
		if (left == null) {
			throw new ParseException("The expression on the left side of null coalescing and safe access operator \"??\" can not be blank", location);
		}
		this.left = left;
		this.right = right;
		this.location = location;
	}

	public Object eval(Scope scope) {
		Ctrl ctrl = scope.getCtrl();
		boolean oldNullSafeValue = ctrl.isNullSafe();

		try {
			ctrl.setNullSafe(true);
			Object ret = left.eval(scope);
			if (ret != null) {
				return ret;
			}
		} finally {
			ctrl.setNullSafe(oldNullSafeValue);
		}

		// right 表达式处于 null safe 区域之外
		return right != null ? right.eval(scope) : null;
	}
}







