/**
 * Copyright (c) 2023 James Zhan 詹波 (zhanbocn@126.com)
 * Aifei Enjoy is licensed under Mulan PSL v2.
 * You can use this software according to the terms and conditions of the Mulan PSL v2.
 * You may obtain a copy of Mulan PSL v2 at:
 *          http://license.coscl.org.cn/MulanPSL2
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PSL v2 for more details.
 */

package cn.aifei.enjoy.expr.ast;

import java.util.List;

import cn.aifei.enjoy.TemplateException;
import cn.aifei.enjoy.stat.Location;
import cn.aifei.enjoy.stat.ParseException;
import cn.aifei.enjoy.stat.Scope;

/**
 * index : expr '[' expr ']'
 *
 * 支持 a[i]、 a[b[i]]、a[i][j]、a[i][j]...[n]
 */
public class Index extends Expr {

	private Expr expr;
	private Expr index;

	public Index(Expr expr, Expr index, Location location) {
		if (expr == null || index == null) {
			throw new ParseException("array/list/map and their index can not be null", location);
		}
		this.expr = expr;
		this.index = index;
		this.location = location;
	}

	@SuppressWarnings("rawtypes")
	public Object eval(Scope scope) {
		Object target = expr.eval(scope);
		if (target == null) {
			if (scope.getCtrl().isNullSafe()) {
				return null;
			}
			throw new TemplateException("The index access operation target can not be null", location);
		}

		Object idx = index.eval(scope);
		if (idx == null) {
			if (scope.getCtrl().isNullSafe()) {
				return null;
			}

			if (target instanceof java.util.Map) {
				// Map 的 key 可以是 null，不能抛异常
			} else {
				throw new TemplateException("The index of list and array can not be null", location);
			}
		}

		if (target instanceof List) {
			if (idx instanceof Integer) {
				return ((List<?>)target).get((Integer)idx);
			}
			throw new TemplateException("The index of list must be integer", location);
		}

		if (target instanceof java.util.Map) {
			return ((java.util.Map)target).get(idx);
		}

		if (target.getClass().isArray()) {
			if (idx instanceof Number) {
				return java.lang.reflect.Array.get(target, ((Number)idx).intValue());
			}
			throw new TemplateException("The index of array must be Number", location);
		}

		throw new TemplateException("Only the list array and map is supported by index access", location);
	}
}




