package cn.afternode.commons.bukkit.report;

import org.bukkit.Bukkit;
import org.bukkit.plugin.Plugin;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

/**
 * A report builder for analysing like CrashReport in Minecraft
 */
public class PluginReport {
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss", Locale.ROOT);
    private static final DateTimeFormatter FILENAME_DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd_HH.mm.ss", Locale.ROOT);

    private final Plugin plugin;

    private final List<IPluginReportElement> elements = new ArrayList<>();

    /**
     * Primary constructor
     * @param plugin Generated by
     */
    public PluginReport(Plugin plugin) {
        this.plugin = plugin;
    }

    /**
     * Append element
     * @param element Element
     * @return This builder
     */
    public PluginReport appendElement(IPluginReportElement element) {
        elements.add(element);
        return this;
    }

    /**
     * Append ServerInfoElement
     * @return This builder
     * @see ServerInfoElement
     */
    public PluginReport withServerInfo() {
        return this.appendElement(new ServerInfoElement(Bukkit.getServer()));
    }

    /**
     * Append PluginInfoElement with default properties
     * @return This builder
     * @see PluginInfoElement
     */
    public PluginReport withPluginInfo() {
        return this.appendElement(new PluginInfoElement(plugin));
    }

    /**
     * Append PluginListElement
     * @return This builder
     * @see PluginListElement
     */
    public PluginReport withPluginList() {
        return this.appendElement(new PluginListElement());
    }

    /**
     * Append PluginReport
     * @param title Title
     * @param throwable Exception
     * @return This builder
     * @see ExceptionElement
     */
    public PluginReport withException(String title, Throwable throwable) {
        return this.appendElement(new ExceptionElement(title, throwable));
    }

    /**
     * Build to string
     * @return Result
     */
    public String build() {
        StringBuilder builder = new StringBuilder();

        builder.append("=== Bukkit Plugin Report ===\n");
        builder.append("Generated at: ").append(DATE_TIME_FORMATTER.format(ZonedDateTime.now())).append("\n");
        builder.append("By: ").append(plugin.getName()).append("\n");

        for (IPluginReportElement element : elements) {
            builder.append("\n\n===").append(element.title()).append("===\n");
            builder.append(element.build()).append("\n");
        }

        return builder.toString();
    }

    /**
     * Save to file
     * @param target Output file path
     */
    public void save(Path target) throws IOException {
        if (Files.notExists(target.getParent()))
            Files.createDirectories(target);

        Files.writeString(target, build());
    }

    /**
     * Save to ~/reports/report-[DateTime]-[PluginName].txt
     * @return Output path
     */
    public Path save() throws IOException {
        Path result = Path.of("reports").resolve("report-%s-%s.txt".formatted(FILENAME_DATE_TIME_FORMATTER.format(ZonedDateTime.now()), plugin.getName()));
        this.save(result);
        return result;
    }

    /**
     * Save to file and print to logger
     * @return Output path
     */
    public Path saveAndLog() throws IOException {
        Path result = save();
        plugin.getLogger().info("Report saved to " + result);
        return result;
    }
}
