package cn.aetherial.openweather.config;

import cn.aetherial.openweather.cache.WeatherCache;
import cn.aetherial.openweather.factory.WeatherDataStrategyFactory;
import cn.aetherial.openweather.properties.OpenWeatherProperties;
import cn.aetherial.openweather.service.OpenWeatherService;
import cn.aetherial.openweather.strategy.DetailedWeatherStrategy;
import cn.aetherial.openweather.strategy.SimpleWeatherStrategy;
import cn.aetherial.openweather.strategy.StandardWeatherStrategy;
import cn.aetherial.openweather.utils.OpenWeatherUtils;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.web.client.RestTemplate;
import java.time.Duration;

@Configuration
@EnableConfigurationProperties(OpenWeatherProperties.class)
public class OpenWeatherAutoConfiguration {

    @Bean
    @ConditionalOnMissingBean
    public SimpleWeatherStrategy simpleWeatherStrategy() {
        return new SimpleWeatherStrategy();
    }
    
    @Bean
    @ConditionalOnMissingBean
    public StandardWeatherStrategy standardWeatherStrategy() {
        return new StandardWeatherStrategy();
    }
    
    @Bean
    @ConditionalOnMissingBean
    public DetailedWeatherStrategy detailedWeatherStrategy(WeatherDetailsConfig config) {
        return new DetailedWeatherStrategy(config);
    }
    
    @Bean
    @ConditionalOnMissingBean
    public WeatherDetailsConfig weatherDetailsConfig() {
        return WeatherDetailsConfig.createDefault();
    }
    
    @Bean
    @ConditionalOnMissingBean
    public WeatherDataStrategyFactory weatherDataStrategyFactory(
            SimpleWeatherStrategy simpleStrategy,
            StandardWeatherStrategy standardStrategy,
            DetailedWeatherStrategy detailedStrategy) {
        return new WeatherDataStrategyFactory(simpleStrategy, standardStrategy, detailedStrategy);
    }

    @Bean
    @ConditionalOnMissingBean
    public ObjectMapper objectMapper() {
        return new ObjectMapper();
    }

    @Bean
    @ConditionalOnMissingBean
    public RestTemplate restTemplate(RestTemplateBuilder restTemplateBuilder, OpenWeatherProperties properties) {
        return restTemplateBuilder
                .setConnectTimeout(Duration.ofMillis(properties.getConnectionTimeout()))
                .setReadTimeout(Duration.ofMillis(properties.getReadTimeout()))
                .build();
    }

    @Bean
    @ConditionalOnMissingBean
    public OpenWeatherService openWeatherService(
            RestTemplate restTemplate,
            OpenWeatherProperties properties,
            WeatherDataStrategyFactory strategyFactory,
            WeatherCache weatherCache,
            ObjectMapper objectMapper) {
        return new OpenWeatherService(restTemplate, properties, strategyFactory, weatherCache, objectMapper);
    }

    @Bean
    @ConditionalOnMissingBean
    public OpenWeatherUtils openWeatherUtils() {
        return new OpenWeatherUtils();
    }
}
