package com.chen.tools.cache;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.io.UnsupportedEncodingException;
import java.util.LinkedHashSet;
import java.util.Set;

/**
 * @author chen
 */
@Service
public class RedisServiceImpl implements RedisService {

    private static final Logger logger = LoggerFactory.getLogger(RedisServiceImpl.class);
    private static String redisCode = "utf-8";

    @Autowired
    private RedisTemplate<String, String> stringRedisTemplate;

    /**
     * 通过key删除
     *
     * @param keys  keys
     * @return long
     */
    public long del(final String... keys) {
        return stringRedisTemplate.execute(new RedisCallback<Long>() {
            public Long doInRedis(RedisConnection connection) throws DataAccessException {
                long result = 0;
                for (int i = 0; i < keys.length; i++) {
                    result = connection.del(keys[i].getBytes());
                }
                return result;
            }
        });
    }

    /**
     * 通过前缀删除
     * @param prex 前缀
     * @return 影响的行数
     */
    public Long delByPrex(final String prex) {
        Set<String> keys = stringRedisTemplate.keys(prex);
        if (!ObjectUtils.isEmpty(keys)) {
            return stringRedisTemplate.delete(keys);
        }
        return 0L;
    }

    /**
     * 添加key value 并且设置存活时间(byte)
     *
     * @param key key
     * @param value value
     * @param liveTime liveTime
     */
    public void set(final byte[] key, final byte[] value, final long liveTime) {
        stringRedisTemplate.execute(new RedisCallback<Long>() {
            public Long doInRedis(RedisConnection connection) throws DataAccessException {
                connection.set(key, value);
                if (liveTime > 0) {
                    connection.expire(key, liveTime);
                }
                return 1L;
            }
        });
    }

    /**
     * 添加key value 并且设置存活时间
     *
     * @param key key
     * @param value value
     * @param liveTime 单位秒
     */
    public void set(String key, String value, long liveTime) {
        try {
            this.set(key.getBytes(redisCode), value.getBytes(redisCode), liveTime);
        } catch (UnsupportedEncodingException e) {
            logger.error("insert into  redis change UTF-8 has Exception", e);
        }
    }

    /**
     * 添加key value
     *
     * @param key key
     * @param value value
     */
    public void set(String key, String value) {
        this.set(key, value, 0L);
    }

    /**
     * 添加key value (字节)(序列化)
     *
     * @param key key
     * @param value value
     */
    public void set(byte[] key, byte[] value) {
        this.set(key, value, 0L);
    }

    /**
     * 获取redis value (String)
     *
     * @param key key
     * @return value
     */
    public String get(final String key) {
        return stringRedisTemplate.execute(new RedisCallback<String>() {
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                try {

                    byte[] bytes = connection.get(key.getBytes());
                    if (bytes == null) {
                        return null;
                    } else {
                        return new String(bytes, redisCode);
                    }

                } catch (Exception e) {
                }
                return "";
            }
        });
    }

    /**
     * 通过正则匹配keys
     *
     * @param pattern 正则
     * @return set String
     */
    public Set<String> Setkeys(String pattern) {
        return stringRedisTemplate.keys(pattern);

    }

    /**
     * 检查key是否已经存在
     *
     * @param key key
     * @return 检查key是否已经存在
     */
    public boolean exists(final String key) {
        return stringRedisTemplate.execute(new RedisCallback<Boolean>() {
            public Boolean doInRedis(RedisConnection connection) throws DataAccessException {
                return connection.exists(key.getBytes());
            }
        });
    }

    /**
     * 清空redis 所有数据
     *
     * @return 清空redis 所有数据
     */
    public String flushDB() {
        return stringRedisTemplate.execute(new RedisCallback<String>() {
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                connection.flushDb();
                return "ok";
            }
        });
    }

    /**
     * 查看redis里有多少数据
     * @return 查看redis里有多少数据
     */
    public long dbSize() {
        return stringRedisTemplate.execute(new RedisCallback<Long>() {
            public Long doInRedis(RedisConnection connection) throws DataAccessException {
                return connection.dbSize();
            }
        });
    }

    /**
     * 检查是否连接成功
     *
     * @return 检查是否连接成功
     */
    public String ping() {
        return stringRedisTemplate.execute(new RedisCallback<String>() {
            public String doInRedis(RedisConnection connection) throws DataAccessException {
                return connection.ping();
            }
        });
    }

    /**
     * 基于redis自增
     * @param key key
     * @return 基于redis自增
     */
    @Override
    public Integer incRedisKey(String key) {
        stringRedisTemplate.execute(new RedisCallback<Integer>() {
            public Integer doInRedis(RedisConnection connection) throws DataAccessException {
                try {
                    connection.incr(key.getBytes("UTF-8"));
                } catch (UnsupportedEncodingException e) {
                    logger.error("setSortSet has error", e);
                }
                return 1;
            }
        });
        return 1;
    }

    /**
     * 往set中添加一个键值对
     *
     * @param key key
     * @param value value
     * @return 往set中添加一个键值对
     */
    public Integer setSortSet(String key, String value) {
        stringRedisTemplate.execute(new RedisCallback<Integer>() {
            public Integer doInRedis(RedisConnection connection) throws DataAccessException {
                try {
                    // 往set中添加元素，已存在的追加 不存在的新建
                    connection.zIncrBy(key.getBytes("UTF-8"), 1, value.getBytes("UTF-8"));
                } catch (UnsupportedEncodingException e) {
                    logger.error("setSortSet has error", e);
                }
                return 1;
            }
        });
        return 1;
    }


    /**
     * 查询前十个最热门的所有车辆
     *
     * @param key
     * @return
     */
    public LinkedHashSet<String> getSortSet(final String key) {
        return stringRedisTemplate.execute(new RedisCallback<LinkedHashSet<String>>() {
            public LinkedHashSet<String> doInRedis(RedisConnection connection) throws DataAccessException {
                try {
                    // 降序查询set中的前十个热门搜索的车辆
                    Set<byte[]> bytesSet = connection.zRevRange(key.getBytes(), 0L, 9L);
                    if (bytesSet == null) {
                        return null;
                    } else {
                        LinkedHashSet<String> set = new LinkedHashSet<String>();
                        for (byte[] bt : bytesSet) {
                            set.add(new String(bt, redisCode));
                        }
                        return set;
                    }
                } catch (UnsupportedEncodingException e) {
                    logger.error("getSortSet has error", e);
                }
                return null;
            }
        });
    }


}
