package ch.stegmaier.java2tex.commands.registry.environments;

import ch.stegmaier.java2tex.commands.BaseCommand;
import ch.stegmaier.java2tex.commands.GenericCommand;
import ch.stegmaier.java2tex.commands.Row;
import ch.stegmaier.java2tex.commands.Tabular;

/**
 * Produce a table, a box consisting of a sequence of horizontal rows. Each row consists of items that are aligned vertically in columns.
 * @see <a href="https://latexref.xyz/tabular.html">tabular</a>
 */
public class Table {

    private Table() {}

    /**
     * creates a simple row helper for the table or tabular environment with column separation and row end
     * @see <a href="https://en.wikibooks.org/wiki/LaTeX/Tables#Basic_examples">Table Examples</a>
     */
    public static Row row() {
        return new Row();
    }

    /**
     * tabular environment
     * @see <a href="https://en.wikibooks.org/wiki/LaTeX/Tables#The_tabular_environment">tabular environment</a>
     */
    public static Tabular tabular() {
        return new Tabular();
    }

    /**
     * A column of left-aligned items.
     */
    public static BaseCommand left() {
        return new BaseCommand("l");
    }

    /**
     * A vertical line the full height and depth of the environment.
     * @return
     */
    public static BaseCommand vLine() {
        return new BaseCommand("|");
    }

    /**
     * A column of right-aligned items.
     */
    public static BaseCommand right() {
        return new BaseCommand("r");
    }

    /**
     * A column of centered items.
     */
    public static BaseCommand center() {
        return new BaseCommand("c");
    }

    /**
     * A columnseparator within a the body of table environment
     */
    public static BaseCommand colSep() {
        return new BaseCommand(" & ");
    }

    /**
     * End of a row in a table environment
     */
    public static BaseCommand lf() {
        return new BaseCommand("\\\\\n");
    }

    /**
     * Draw a horizontal line the width of the enclosing tabular or array environment. It’s most commonly used to draw a line at the top, bottom, and between the rows of a table.
     *
     * @see <a href="https://latexref.xyz/_005chline.html">hline</a>
     */
    public static GenericCommand hline() {
        return new GenericCommand("hline").autoNewline();
    }
}
