package ch.stegmaier.java2tex.commands.registry.environments;

import ch.stegmaier.java2tex.commands.BaseCommand;
import ch.stegmaier.java2tex.commands.GenericCommand;

/**
 * Some typographic elements, such as figures and tables, cannot be broken across pages. They must be typeset outside of the normal flow of text, for instance floating to the top of a later page.
 * @see <a href="https://latexref.xyz/Floats.html">Floats</a>
 * @see <a href="https://latexref.xyz/figure.html">figure</a>
 * @see <a href="https://latexref.xyz/table.html">table</a>
 */
public class Floats {

    private Floats() {}

    /**
     * Make a caption for a floating environment, such as a figure or table environment
     * @see <a href="https://latexref.xyz/_005ccaption.html">caption command</a>
     */
    public static GenericCommand caption() {
        return new GenericCommand("caption").emptyArgs().autoNewline();
    }

    /**
     * Places the float at precisely the location in the LaTeX code. Requires the float package, i.e., \\usepackage{float}.
     */
    public static BaseCommand exactHere() {
        return new BaseCommand("H");
    }

    /**
     * Override internal parameters LaTeX uses for determining "good" float positions.
     */
    public static BaseCommand override() {
        return new BaseCommand("!");
    }

    /**
     * Put on a special page for floats only.
     */
    public static BaseCommand page() {
        return new BaseCommand("p");
    }

    /**
     * Position at the bottom of the page.
     */
    public static BaseCommand bottom() {
        return new BaseCommand("b");
    }

    /**
     * Position at the top of the page.
     */
    public static BaseCommand top() {
        return new BaseCommand("t");
    }

    /**
     * Place the float here, i.e., approximately at the same point it occurs in the source text (however, not exactly at the spot)
     */
    public static BaseCommand here() {
        return new BaseCommand("h");
    }
}
