package ch.stegmaier.java2tex.commands;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

public class GenericCommand<X extends GenericCommand> extends BaseCommand<X> {

    private boolean autoNewline;
    private boolean star;
    private boolean noFormat;
    private boolean emptyArgs;
    private boolean notPrefix;
    private LinkedList<String> mandatoryArguments = new LinkedList<>();
    private LinkedList<String> optionalArguments = new LinkedList<>();
    private StringBuilder content = new StringBuilder();

    public GenericCommand(final String commandName) {
        super(commandName);
    }

    public X autoNewline() {
        this.autoNewline = true;
        return getThis();
    }

    public X noFormat() {
        this.noFormat = true;
        return getThis();
    }

    public X noPrefix() {
        this.notPrefix = true;
        return getThis();
    }

    public X emptyArgs() {
        this.emptyArgs = true;
        return getThis();
    }

    public void clear() {
        this.mandatoryArguments.clear();
        this.optionalArguments.clear();
        content = new StringBuilder();
    }

    public X argument(Integer argument) {
        mandatoryArguments.add(argument.toString());
        return getThis();
    }

    public X argument(String argument) {
        mandatoryArguments.add(noFormat ? argument : formatValue(argument));
        return getThis();
    }

    public X argument(BaseCommand argument) {
        mandatoryArguments.add(argument.toString());
        return getThis();
    }

    public X argument(BaseCommand... arguments) {
        argument(Arrays.asList(arguments).stream().map(BaseCommand::toString).collect(Collectors.joining()));
        return getThis();
    }

    public X arguments(BaseCommand... arguments) {
        Arrays.asList(arguments).forEach(arg -> argument(arg));
        return getThis();
    }

    public X append(BaseCommand command) {
        content.append(command);
        return getThis();
    }

    public X append(float value) {
        return append("" + value, false);
    }

    public X append(String text) {
        return append(text, false);
    }

    protected X append(String text, boolean skipSpace) {
        if (content.length()==0 && !StringUtils.isEmpty(getCommand()) && !skipSpace) {
            content.append(" ");
        }
        content.append(text);
        return getThis();
    }

    public X option(String option) {
        optionalArguments.add(option);
        return getThis();
    }

    public X option(BaseCommand option) {
        optionalArguments.add(option.toString());
        return getThis();
    }

    public X option(BaseCommand... options) {
        optionalArguments.add(Arrays.asList(options).stream().map(BaseCommand::toString).collect(Collectors.joining()));
        return getThis();
    }

    public X star() {
        this.star = true;
        return getThis();
    }

    protected String buildName() {
        return StringUtils.isEmpty(getCommand()) ? "" : ((notPrefix ? "" : "\\") + getCommand() + (star ? "*" : ""));
    }

    protected String appendNewLine() {
        return autoNewline ? "\n" : "";
    }

    protected String buildOptional() {
        return optionalArguments.stream().collect(Collectors.joining("][", "[", "]")).replace("[]", "");
    }

    protected String getContent() {
        return content.toString();
    }

    protected String buildMandatory() {
        var args = mandatoryArguments.stream().collect(Collectors.joining("}{", "{", "}"));
        if (emptyArgs) {
            return args;
        }
        return args.replace("{}", "");
    }

    @Override
    public String toString() {
        var sb = new StringBuilder()
                .append(buildName())
                .append(buildOptional())
                .append(buildMandatory())
                .append(appendNewLine())
                .append(getContent());
        return sb.toString();
    }

}
