package cc.xbyter.code.generator.engine;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import cc.xbyter.code.generator.config.ConstVal;
import cc.xbyter.code.generator.config.GlobalConfig;
import cc.xbyter.code.generator.config.TemplateConfig;
import cc.xbyter.code.generator.config.builder.ConfigBuilder;
import cc.xbyter.code.generator.config.po.TableInfo;
import cc.xbyter.code.generator.config.rules.ProjectType;
import cc.xbyter.code.generator.config.rules.StringPool;
import cc.xbyter.code.generator.config.rules.StringUtils;
import lombok.Getter;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.apache.velocity.app.VelocityEngine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * @author cc
 */
public abstract class AbstractTemplateEngine {
    protected static final Logger logger = LoggerFactory.getLogger("TemplateEngine");
    private static final String DOT_VM = ".vm";


    private VelocityEngine velocityEngine;
    @Getter
    ConfigBuilder configBuilder;

    public abstract AbstractTemplateEngine mkdirs();

    public abstract AbstractTemplateEngine batchOutPut() throws Exception ;

    public AbstractTemplateEngine init(ConfigBuilder ConfigBuilder) {
        this.configBuilder = ConfigBuilder;
        if (null == velocityEngine) {
            Properties p = new Properties();
            p.setProperty(ConstVal.VM_LOAD_PATH_KEY, ConstVal.VM_LOAD_PATH_VALUE);
            p.setProperty(Velocity.FILE_RESOURCE_LOADER_PATH, StringPool.EMPTY);
            p.setProperty(Velocity.ENCODING_DEFAULT, ConstVal.UTF8);
            p.setProperty(Velocity.INPUT_ENCODING, ConstVal.UTF8);
            p.setProperty("file.resource.loader.unicode", StringPool.TRUE);
            velocityEngine = new VelocityEngine(p);
        }
        return this;
    }

    public Map<String, Object> getObjectMap(TableInfo tableInfo) {
        Map<String, Object> objectMap = new HashMap<>();
        ConfigBuilder config = getConfigBuilder();

        objectMap.put("config", config);
        GlobalConfig globalConfig = config.getGlobalConfig();
        objectMap.put("author", globalConfig.getAuthor());
        objectMap.put("swagger", globalConfig.isSwagger());
        objectMap.put("projectClassName", StringUtils.toCamelUpperCase(globalConfig.getProjectName()));
        objectMap.put("projectPackName", configBuilder.getPackageConfig().getParent());
        objectMap.put("mapperScanPackName", configBuilder.getPackageConfig().getColaMapperScanPackage());
        objectMap.put("schemaName", "");
        objectMap.put("date", new SimpleDateFormat("yyyy-MM-dd").format(new Date()));
        objectMap.put("table", tableInfo);
        objectMap.put("entity", tableInfo.getEntityName());
        if (globalConfig.getProjectType().equals(ProjectType.COLA)) {
            objectMap.put("entityClassName", tableInfo.getEntityName() + "DB");
            objectMap.put("superEntityClass", StringUtils.getClassName(globalConfig.getColaSuperEntityClass()));
            objectMap.put("superEntityImport", globalConfig.getColaSuperEntityClass());
            objectMap.put("superDomainClass", StringUtils.getClassName(globalConfig.getColaSuperDomainClass()));
            objectMap.put("domainClassName", tableInfo.getEntityName() + "Entity");
        } else {
            objectMap.put("entityClassName", tableInfo.getEntityName());
            objectMap.put("superDomainClass", "");
        }
        objectMap.put("entityLombokModel", config.getStrategyConfig().isEntityLombokModel());
        objectMap.put("chainModel", config.getStrategyConfig().isChainModel());
        objectMap.put("package", config.getPackageInfo());

        Set<String> applicationPackages = new HashSet<>();
        applicationPackages.add("org.mybatis.spring.annotation.MapperScan");
        applicationPackages.add("org.springframework.boot.SpringApplication");
        applicationPackages.add("org.springframework.boot.autoconfigure.SpringBootApplication");
        objectMap.put("applicationPackages", applicationPackages);

        return objectMap;
    }

    public void writer(Map<String, Object> objectMap, String templatePath, String outputFile) throws Exception {
        Template template = velocityEngine.getTemplate(templatePath, ConstVal.UTF8);
        try (FileOutputStream fos = new FileOutputStream(outputFile);
             OutputStreamWriter ow = new OutputStreamWriter(fos, ConstVal.UTF8);
             BufferedWriter writer = new BufferedWriter(ow)) {
            template.merge(new VelocityContext(objectMap), writer);
        }
        logger.debug("模板:" + templatePath + ";  文件:" + outputFile);
    }

    public String templateFilePath(String filePath) {
        if (null == filePath || filePath.contains(DOT_VM)) {
            return filePath;
        }
        return filePath + DOT_VM;
    }
}
