/*
 * Copyright (c) 2011-2023, baomidou (jobob@qq.com).
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cc.xbyter.code.generator.config.po;

import cc.xbyter.code.generator.config.rules.ProjectType;
import cn.hutool.core.collection.CollUtil;
import cc.xbyter.code.generator.config.GlobalConfig;
import cc.xbyter.code.generator.config.PackageConfig;
import cc.xbyter.code.generator.config.StrategyConfig;
import cc.xbyter.code.generator.config.builder.ConfigBuilder;
import cc.xbyter.code.generator.config.rules.NamingStrategy;
import cc.xbyter.code.generator.config.rules.StringUtils;
import lombok.Data;
import lombok.Getter;
import lombok.experimental.Accessors;
import org.jetbrains.annotations.NotNull;

import java.util.*;
import java.util.stream.Collectors;


/**
 * 表信息，关联到当前字段信息
 *
 * @author YangHu, lanjerry
 * @since 2016/8/30
 */
@Data
@Accessors(chain = true)
public class TableInfo {
    /**
     * 策略配置
     */
    private final StrategyConfig strategyConfig;
    /**
     * 全局配置信息
     */
    private final GlobalConfig globalConfig;
    /**
     * 包信息
     */
    PackageConfig packageConfig;

    /**
     * 是否转换
     */
    @Getter
    private boolean convert;

    /**
     * 表名称
     */
    @Getter
    private String name;

    /**
     * 表注释
     */
    @Getter
    private String comment;

    /**
     * 实体名称:首字母大写
     */
    private String entityName;

    /**
     * 变量名称:首字母小写
     */
    private String varName;
    /**
     * mapper名称
     */
    @Getter
    private String mapperName;

    /**
     * xml名称
     */
    @Getter
    private String xmlName;

    /**
     * service名称
     */
    @Getter
    private String serviceName;

    /**
     * serviceImpl名称
     */
    @Getter
    private String serviceImplName;

    /**
     * controller名称
     */
    @Getter
    private String controllerName;

    /**
     * 表字段
     */
    private List<TableField> fields = new ArrayList<>();

    /**
     * 是否有主键
     */
    @Getter
    private boolean havePrimaryKey;

    /**
     * 公共字段
     */
    private final List<TableField> commonFields = new ArrayList<>();

    /**
     * 字段名称集
     */
    private String fieldNames;

    // region 需导入包的信息
    /**
     * 通用包导入信息
     */
    private final Set<String> importPackages = new TreeSet<>();
    /**
     * 领域类需要特殊导入的包信息
     */
    private final Set<String> importDomainPackages = new TreeSet<>();
    /**
     * 数据库实体类需要特殊导入的包信息
     */
    private final Set<String> importEntityPackages = new TreeSet<>();
    /**
     * 数据库Mapper类需要特殊导入的包信息
     */
    private final Set<String> importMapperPackages = new TreeSet<>();
    /**
     * Gateway需要特殊导入的包信息
     */
    private final Set<String> importGatewayPackages = new TreeSet<>();
    /**
     * Gateway需要特殊导入的包信息
     */
    private final Set<String> importGatewayImplPackages = new TreeSet<>();
    /**
     * PageQuery需要特殊导入的包信息
     */
    private final Set<String> importPageQueryPackages = new TreeSet<>();
    /**
     * EditCmd需要特殊导入的包信息
     */
    private final Set<String> importEditCmdPackages = new TreeSet<>();
    /**
     * Cola Service需要特殊导入的包信息
     */
    private final Set<String> importColaServicePackages = new TreeSet<>();
    /**
     * Cola ServiceImpl需要特殊导入的包信息
     */
    private final Set<String> importColaServiceImplPackages = new TreeSet<>();

    /**
     * Cola Controller需要特殊导入的包信息
     */
    private final Set<String> importColaControllerPackages = new TreeSet<>();

    // endregion

    /**
     * 构造方法
     *
     * @param configBuilder 配置构建
     * @param name          表名
     * @since 3.5.0
     */
    public TableInfo(@NotNull ConfigBuilder configBuilder, @NotNull String name) {
        this.strategyConfig = configBuilder.getStrategyConfig();
        this.globalConfig = configBuilder.getGlobalConfig();
        this.packageConfig = configBuilder.getPackageConfig();
        this.name = name;
        initImportPackages();
    }


    public TableInfo(@NotNull ConfigBuilder configBuilder, @NotNull String name, String comment) {
        this.strategyConfig = configBuilder.getStrategyConfig();
        this.globalConfig = configBuilder.getGlobalConfig();
        this.packageConfig = configBuilder.getPackageConfig();
        this.name = name;
        this.comment = comment;
        initImportPackages();
    }

    /**
     * 初始化生成文件的引入包
     */
    private void initImportPackages() {
        if (globalConfig.getProjectType().equals(ProjectType.COLA)) {
            initImportColaPackages();
        } else {
            initImportMvcPackages();
        }
    }


    public String getEntityPath() {
        return entityName.substring(0, 1).toLowerCase() + entityName.substring(1);
    }


    /**
     * @param pkgs 包空间
     * @return this
     * @since 3.5.0
     */
    public TableInfo addImportPackages(@NotNull String... pkgs) {
        return addImportPackages(Arrays.asList(pkgs));
    }

    public TableInfo addImportPackages(@NotNull List<String> pkgList) {
        importPackages.addAll(pkgList);
        return this;
    }

    public void setFields(List<TableField> list) {
        this.fields = list;
        if (CollUtil.isNotEmpty(fields)) {
            // 收集导入包信息
            for (TableField field : fields) {
                if (null != field.getColumnType() && null != field.getColumnType().getPkg()) {
                    importPackages.add(field.getColumnType().getPkg());
                }
                //if (field.isKeyFlag()) {
                //    // 主键
                //    if (field.isKeyIdentityFlag()) {
                //        importPackages.add("com.baomidou.mybatisplus.annotation.TableId");
                //    }
                //    // 自增
                //    if (field.isKeyIdentityFlag()) {
                //        importPackages.add("com.baomidou.mybatisplus.annotation.IdType");
                //    }
                //}
            }
        }
    }

    public void setEntityName(String entityName) {
        this.entityName = entityName;
        this.varName = NamingStrategy.lowerFirst(this.entityName);
        importGatewayPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaDomain(), globalConfig.getDomainPackageName(), this.getEntityName() + "Entity"));
        importGatewayPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaData(), this.getEntityName() + "DB"));

        importMapperPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaData(), this.getEntityName() + "DB"));

        importGatewayImplPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaDomain(), globalConfig.getDomainPackageName(), this.getEntityName() + "Entity"));
        importGatewayImplPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaData(), this.getEntityName() + "DB"));
        importGatewayImplPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaMapper(), this.getEntityName() + "Mapper"));
        importGatewayImplPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaGateway(), this.getEntityName() + "Gateway"));
        importGatewayImplPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaPageQuery(), this.getEntityName() + "PageQuery"));

        importColaServicePackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaPageQuery(), this.getEntityName() + "PageQuery"));
        importColaServicePackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaDetailVO(), this.getEntityName() + "PageItemVO"));
        importColaServicePackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaDetailVO(), this.getEntityName() + "DetailVO"));
        importColaServicePackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaEditCmd(), this.getEntityName() + "EditCmd"));

        importColaServiceImplPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaPageQuery(), this.getEntityName() + "PageQuery"));
        importColaServiceImplPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaDetailVO(), this.getEntityName() + "PageItemVO"));
        importColaServiceImplPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaDetailVO(), this.getEntityName() + "DetailVO"));
        importColaServiceImplPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaEditCmd(), this.getEntityName() + "EditCmd"));
        importColaServiceImplPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaGateway(), this.getEntityName() + "Gateway"));
        importColaServiceImplPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaService(), this.getEntityName() + "Service"));
        importColaServiceImplPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaDomain(), globalConfig.getDomainPackageName(), this.getEntityName() + "Entity"));

        importColaControllerPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaService(), this.getEntityName() + "Service"));
        importColaControllerPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaEditCmd(), this.getEntityName() + "EditCmd"));
        importColaControllerPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaDetailVO(), this.getEntityName() + "DetailVO"));
        importColaControllerPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaDetailVO(), this.getEntityName() + "PageItemVO"));
        importColaControllerPackages.add(StringUtils.buildPackageName(packageConfig.getParent(),
                packageConfig.getColaPageQuery(), this.getEntityName() + "PageQuery"));
    }


    void initImportColaPackages() {
        // region 初始化实体类的引入包
        addToSet(importEntityPackages,
                "com.baomidou.mybatisplus.annotation.*",
                "lombok.EqualsAndHashCode",
                "com.baomidou.mybatisplus.annotation.IdType",
                "com.baomidou.mybatisplus.annotation.TableId",
                "com.baomidou.mybatisplus.annotation.TableName"
        );
        // endregion

        // region 初始化领域类的引入包
        addToSet(importDomainPackages, globalConfig.getColaSuperDomainClass());
        // endregion

        // region 初始化Mapper类的引入包
        addToSet(importMapperPackages,
                "com.baomidou.mybatisplus.core.mapper.BaseMapper",
                "org.apache.ibatis.annotations.Mapper");
        // endregion

        // region 初始化Gateway类的引入包
        addToSet(importGatewayPackages, "com.linkduoo.brand.core.mybatisplus.IGateway");
        // endregion

        // region 初始化GatewayImpl类的引入包
        addToSet(importGatewayImplPackages,
                "com.baomidou.mybatisplus.core.conditions.query.QueryWrapper",
                "com.linkduoo.brand.core.base.dto.BasePageQuery",
                "com.linkduoo.brand.core.mybatisplus.GatewayImpl",
                "org.springframework.stereotype.Component");

        // endregion

        // region 初始化PageQuery类的引入包
        addToSet(importPageQueryPackages, "com.linkduoo.brand.core.base.dto.BasePageQuery");
        // endregion

        // region 初始化EditCmd类的引入包
        addToSet(importEditCmdPackages, "com.linkduoo.brand.core.base.dto.BaseEditCmd");
        // endregion

        // region 初始化Service类的引入包
        addToSet(importColaServicePackages,
                "com.linkduoo.brand.core.base.dto.BaseIdCmd",
                "com.linkduoo.brand.core.base.vo.BaseEditVO",
                "com.linkduoo.brand.core.base.vo.PageVO");
        // endregion

        // region 初始化ServiceImpl类的引入包
        addToSet(importColaServiceImplPackages,
                "cn.hutool.core.bean.BeanUtil",
                "com.linkduoo.brand.core.base.dto.BaseIdCmd",
                "com.linkduoo.brand.core.base.vo.BaseEditVO",
                "com.linkduoo.brand.core.base.vo.PageVO",
                "lombok.AllArgsConstructor",
                "lombok.extern.slf4j.Slf4j",
                "org.springframework.stereotype.Service");
        // endregion

        // region 初始化Controller类的引入包
        addToSet(importColaControllerPackages,
                "com.linkduoo.brand.core.base.Result",
                "com.linkduoo.brand.core.base.dto.BaseIdCmd",
                "com.linkduoo.brand.core.base.dto.BaseSetStatusCmd",
                "com.linkduoo.brand.core.base.vo.BaseEditVO",
                "com.linkduoo.brand.core.base.vo.PageVO",
                "io.swagger.annotations.Api",
                "io.swagger.annotations.ApiOperation",
                "lombok.AllArgsConstructor",
                "org.springframework.validation.annotation.Validated",
                "org.springframework.web.bind.annotation.PostMapping",
                "org.springframework.web.bind.annotation.RequestBody",
                "org.springframework.web.bind.annotation.RequestMapping",
                "org.springframework.web.bind.annotation.RestController");
        // endregion
    }

    void initImportMvcPackages() {
        throw new UnsupportedOperationException("Method not implemented yet");
    }

    void addToSet(Set<String> set, String... pkgs) {
        set.addAll(Arrays.asList(pkgs));
    }
}
