package cc.xbyter.code.generator.config.builder;

import cc.xbyter.code.generator.config.rules.*;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import cc.xbyter.code.generator.config.*;
import cc.xbyter.code.generator.config.po.TableField;
import cc.xbyter.code.generator.config.po.TableInfo;
import lombok.Getter;
import org.apache.commons.lang3.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author cc
 */
public class ConfigBuilder {
    private static final Logger log = LoggerFactory.getLogger(ConfigBuilder.class);
    // region 属性

    /**
     * 全局配置
     */
    @Getter
    private GlobalConfig globalConfig;
    /**
     * 策略配置信息
     */
    @Getter
    private StrategyConfig strategyConfig;
    /**
     * 模版配置信息
     */
    @Getter
    private TemplateConfig templateConfig;
    /**
     * 程序包配置信息
     */
    @Getter
    private PackageConfig packageConfig;
    /**
     * 数据库配置信息
     */
    @Getter
    private DataSourceConfig dataSourceConfig;
    /**
     * 数据库表信息
     */
    private final List<TableInfo> tableInfoList = new ArrayList<>();
    /**
     * 需要生成的目录信息
     */
    @Getter
    private final Map<String, String> pathInfo = new HashMap<>();
    /**
     * 需要创建的包目录信息
     */
    @Getter
    private final Map<String, String> packageInfo = new HashMap<>();

    // endregion

    // region 构造方法
    public ConfigBuilder(PackageConfig packageConfig, GlobalConfig globalConfig, StrategyConfig strategyConfig,
                         DataSourceConfig dataSourceConfig, TemplateConfig templateConfig) {
        this.packageConfig = packageConfig;
        this.globalConfig = globalConfig;
        this.strategyConfig = strategyConfig;
        this.dataSourceConfig = dataSourceConfig;
        this.templateConfig = Optional.ofNullable(templateConfig).orElse(new TemplateConfig());

        handlerPackage(globalConfig.getOutputDir(), packageConfig);
    }

    // endregion

    // region 返回读取的表信息
    public List<TableInfo> getTableInfoList() {
        if (tableInfoList.isEmpty()) {
            List<TableInfo> tableInfos = this.queryTables();
            if (!tableInfos.isEmpty()) {
                this.tableInfoList.addAll(tableInfos);
            }
        }
        return tableInfoList;
    }

    // endregion

    // region 处理目录
    private void handlerPackage(String outputDir, PackageConfig config)  {
        if(globalConfig.getProjectType().equals(ProjectType.COLA)){
            handlerColaPackage(outputDir, config);
        }else{
            throw new RuntimeException("未实现");
        }
    }

    private void handlerColaPackage(String outputDir, PackageConfig config) {
        // region 包信息
        // 包信息
        packageInfo.put(ConstVal.COLA_ENTITY, StringUtils.joinPackage(config.getParent(), config.getColaData()));
        packageInfo.put(ConstVal.COLA_MAPPER, StringUtils.joinPackage(config.getParent(), config.getColaMapper()));
        packageInfo.put(ConstVal.COLA_DOMAIN, StringUtils.joinPackage(config.getParent(), config.getColaDomain()) + "." + globalConfig.getDomainPackageName());
        packageInfo.put(ConstVal.COLA_GATEWAY, StringUtils.joinPackage(config.getParent(), config.getColaGateway()));
        packageInfo.put(ConstVal.COLA_GATEWAY_IMPL, StringUtils.joinPackage(config.getParent(), config.getColaGatewayImpl()));
        packageInfo.put(ConstVal.COLA_PAGE_QUERY, StringUtils.joinPackage(config.getParent(), config.getColaPageQuery()));
        packageInfo.put(ConstVal.COLA_EDIT_CMD, StringUtils.joinPackage(config.getParent(), config.getColaEditCmd()));
        packageInfo.put(ConstVal.COLA_DETAIL_VO, StringUtils.joinPackage(config.getParent(), config.getColaDetailVO()));
        packageInfo.put(ConstVal.COLA_PAGE_ITEM_VO, StringUtils.joinPackage(config.getParent(), config.getColaDetailVO()));
        packageInfo.put(ConstVal.COLA_SERVICE, StringUtils.joinPackage(config.getParent(), config.getColaService()));
        packageInfo.put(ConstVal.COLA_SERVICE_IMPL, StringUtils.joinPackage(config.getParent(), config.getColaServiceImpl()));
        packageInfo.put(ConstVal.COLA_CONTROLLER, StringUtils.joinPackage(config.getParent(), config.getColaController()));
        packageInfo.put(ConstVal.COLA_START_APPLICATION, StringUtils.joinPackage(config.getParent(), config.getColaStartApplication()));
        packageInfo.put(ConstVal.PROJECT_NAME, globalConfig.getProjectName());

        // endregion

        // region 工程目录
        pathInfo.put(ConstVal.PROJECT_PATH, StringUtils.joinDir(outputDir, globalConfig.getProjectName()));
        pathInfo.put(ConstVal.COLA_ADAPTER_PROJECT, StringUtils.joinDir(outputDir, globalConfig.getProjectName(),
                globalConfig.getProjectName() + "-adapter"));
        pathInfo.put(ConstVal.COLA_CLIENT_PROJECT, StringUtils.joinDir(outputDir, globalConfig.getProjectName(),
                globalConfig.getProjectName() + "-client"));
        pathInfo.put(ConstVal.COLA_APP_PROJECT, StringUtils.joinDir(outputDir, globalConfig.getProjectName(),
                globalConfig.getProjectName() + "-app"));
        pathInfo.put(ConstVal.COLA_DOMAIN_PROJECT, StringUtils.joinDir(outputDir, globalConfig.getProjectName(),
                globalConfig.getProjectName() + "-domain"));
        pathInfo.put(ConstVal.COLA_INFRASTRUCTURE_PROJECT, StringUtils.joinDir(outputDir, globalConfig.getProjectName(),
                globalConfig.getProjectName() + "-infrastructure"));
        pathInfo.put(ConstVal.COLA_DATA_PROJECT, StringUtils.joinDir(outputDir, globalConfig.getProjectName(),
                globalConfig.getProjectName() + "-data"));
        pathInfo.put(ConstVal.COLA_START_PROJECT, StringUtils.joinDir(outputDir, globalConfig.getProjectName(),
                globalConfig.getProjectName() + "-start"));
        // endregion

        // region adapter目录
        pathInfo.put(ConstVal.COLA_CONTROLLER_PATH, StringUtils.joinDir(pathInfo.get(ConstVal.COLA_ADAPTER_PROJECT), ConstVal.MAIN_PATH,
                config.getParentPath(), "web"));
        // endregion

        // region client目录
        pathInfo.put(ConstVal.COLA_SERVICE_PATH, StringUtils.joinDir(pathInfo.get(ConstVal.COLA_CLIENT_PROJECT), ConstVal.MAIN_PATH,
                config.getParentPath(), "client/api"));
        pathInfo.put(ConstVal.COLA_DTO_QUERY_PATH, StringUtils.joinDir(pathInfo.get(ConstVal.COLA_CLIENT_PROJECT), ConstVal.MAIN_PATH,
                config.getParentPath(), "client/dto/query"));
        pathInfo.put(ConstVal.COLA_DTO_CMD_PATH, StringUtils.joinDir(pathInfo.get(ConstVal.COLA_CLIENT_PROJECT), ConstVal.MAIN_PATH,
                config.getParentPath(), "client/dto/cmd"));
        pathInfo.put(ConstVal.COLA_DTO_DETAIL_VO_PATH, StringUtils.joinDir(pathInfo.get(ConstVal.COLA_CLIENT_PROJECT), ConstVal.MAIN_PATH,
                config.getParentPath(), "client/dto/data"));
        // endregion

        // region app目录
        pathInfo.put(ConstVal.COLA_SERVICE_IMPL_PATH, StringUtils.joinDir(pathInfo.get(ConstVal.COLA_APP_PROJECT), ConstVal.MAIN_PATH,
                config.getParentPath(), "app/service"));
        // endregion

        // region domain目录
        pathInfo.put(ConstVal.COLA_GATEWAY_PATH, StringUtils.joinDir(pathInfo.get(ConstVal.COLA_DOMAIN_PROJECT), ConstVal.MAIN_PATH,
                config.getParentPath(), "domain/gateway"));
        pathInfo.put(ConstVal.COLA_DOMAIN_PATH, StringUtils.joinDir(pathInfo.get(ConstVal.COLA_DOMAIN_PROJECT), ConstVal.MAIN_PATH,
                config.getParentPath(), "domain/"));
        pathInfo.put(ConstVal.COLA_DOMAIN_CHILD_PATH, StringUtils.joinDir(pathInfo.get(ConstVal.COLA_DOMAIN_PROJECT), ConstVal.MAIN_PATH,
                config.getParentPath(), "domain/" + globalConfig.getDomainPackageName()));
        // endregion

        // region infrastructure目录
        pathInfo.put(ConstVal.COLA_GATEWAY_IMPL_PATH, StringUtils.joinDir(pathInfo.get(ConstVal.COLA_INFRASTRUCTURE_PROJECT), ConstVal.MAIN_PATH,
                config.getParentPath(), "infrastructure/gateway/impl"));
        pathInfo.put(ConstVal.COLA_MAPPER_PATH, StringUtils.joinDir(pathInfo.get(ConstVal.COLA_INFRASTRUCTURE_PROJECT), ConstVal.MAIN_PATH,
                config.getParentPath(), "infrastructure/gateway/impl/mapper"));
        // endregion

        // region data目录
        pathInfo.put(ConstVal.COLA_DATA_PROJECT, StringUtils.joinDir(pathInfo.get(ConstVal.COLA_DATA_PROJECT), ConstVal.MAIN_PATH,
                config.getParentPath(), "data/dataobject"));
        // endregion

        // region start目录
        pathInfo.put(ConstVal.COLA_START_PROJECT, StringUtils.joinDir(pathInfo.get(ConstVal.COLA_START_PROJECT), ConstVal.MAIN_PATH,
                config.getParentPath(), packageConfig.getColaStartApplication()));

        // endregion

        packageInfo.putAll(pathInfo);
    }

    // endregion

    // region 处理名字

    private String processName(String name, NamingStrategy strategy, String[] prefix) {
        String propertyName;
        if (ArrayUtils.isNotEmpty(prefix)) {
            if (strategy == NamingStrategy.underline_to_camel) {
                // 删除前缀、下划线转驼峰
                propertyName = NamingStrategy.removePrefixAndCamel(name, prefix);
            } else {
                // 删除前缀
                propertyName = NamingStrategy.removePrefix(name, prefix);
            }
        } else if (strategy == NamingStrategy.underline_to_camel) {
            // 下划线转驼峰
            propertyName = NamingStrategy.underlineToCamel(name);
        } else {
            // 不处理
            propertyName = name;
        }
        return propertyName;
    }

    // endregion

    // region 查询mysql的表信息
    List<TableInfo> queryTables() {
        if (CollUtil.isEmpty(this.strategyConfig.getIncludeList())) {
            throw new RuntimeException("请选择要生成的表");
        }
        List<TableInfo> tableList = new ArrayList<>();
        String sqlTable = this.getSqlTable();
        try (Connection conn = this.dataSourceConfig.getConn()) {
            try (PreparedStatement preparedStatement = conn.prepareStatement(sqlTable);
                 ResultSet resultSet = preparedStatement.executeQuery()) {
                while (resultSet.next()) {
                    String tableName = resultSet.getString("Name");
                    String comment = resultSet.getString("Comment");
                    // 处理每一行数据
                    TableInfo tableInfo = new TableInfo(this, tableName, comment);
                    tableInfo.setEntityName(NamingStrategy.capitalFirst(
                            processName(tableInfo.getName(), globalConfig.getTableNaming(), strategyConfig.getTablePrefix())));


                    System.out.println("tableName: " + tableName + ", comment: " + comment);
                    tableList.add(tableInfo);
                }
            }

            if (CollUtil.isNotEmpty(tableList)) {
                for (TableInfo tableInfo : tableList) {
                    List<TableField> tableFieldList = this.getTableField(conn, tableInfo.getName());
                    tableInfo.setFields(tableFieldList);
                }
            }
        } catch (Exception e) {
            log.error("===========error=========");
            log.error("连接数据库失败：" + e);
            log.error("=========================");
        }


        return tableList;
    }

    String getSqlTable() {
        return "show table status WHERE 1=1  AND NAME IN (" +
                this.strategyConfig.getIncludeList().stream()
                        .map(tb -> "'" + tb + "'")
                        .collect(Collectors.joining(",")) +
                ")";
    }

    // endregion

    // region 查询mysql的表的字端信息
    List<TableField> getTableField(Connection conn, String tableName) {
        String sqlColumns = StrUtil.format("show full fields from `{}`", tableName);

        List<TableField> tableFieldList = new ArrayList<>();
        try (PreparedStatement preparedStatement = conn.prepareStatement(sqlColumns);
             ResultSet resultSet = preparedStatement.executeQuery()) {
            while (resultSet.next()) {
                String fieldName = resultSet.getString("Field");
                String fieldType = resultSet.getString("Type");
                String fieldKey = resultSet.getString("Key");
                String fieldComment = resultSet.getString("Comment");
                String extra = resultSet.getString("Extra");
                //auto_increment

                TableField tableField = new TableField();
                tableField.setName(fieldName).setColumnName(fieldName).setComment(fieldComment)
                        .setType(fieldType).setKeyFlag("PRI".equals(fieldKey));
                tableField.setKeyIdentityFlag(StrUtil.isNotEmpty(extra) && extra.contains("auto_increment"));
                tableField.setColumnType(TypeConverts.convert(globalConfig, fieldType));
                tableField.setPropertyName(processName(fieldName, globalConfig.getColumnNaming(), this.strategyConfig.getFieldPrefix()));

                tableFieldList.add(tableField);

                System.out.println("fieldName: " + fieldName + ", fieldComment: " + fieldComment);
            }
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
        return tableFieldList;
    }

    // endregion

    // region getter+setter
    public ConfigBuilder setStrategyConfig(StrategyConfig strategyConfig) {
        this.strategyConfig = strategyConfig;
        return this;
    }

    public ConfigBuilder setPackageConfig(PackageConfig packageConfig) {
        this.packageConfig = packageConfig;
        return this;
    }

    public ConfigBuilder setGlobalConfig(GlobalConfig globalConfig) {
        this.globalConfig = globalConfig;
        return this;
    }

    public ConfigBuilder setTemplateConfig(TemplateConfig templateConfig) {
        this.templateConfig = templateConfig;
        return this;
    }

    public ConfigBuilder setDataSourceConfig(DataSourceConfig dataSourceConfig) {
        this.dataSourceConfig = dataSourceConfig;
        return this;
    }
    // endregion

}
