/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */
package aws.sdk.kotlin.runtime.protocol.xml

import aws.sdk.kotlin.runtime.InternalSdkApi
import aws.sdk.kotlin.runtime.http.ErrorDetails
import aws.smithy.kotlin.runtime.serde.*
import aws.smithy.kotlin.runtime.serde.xml.XmlDeserializer
import aws.smithy.kotlin.runtime.serde.xml.XmlSerialName

/**
 * Provides access to specific values regardless of message form
 */
internal interface RestXmlErrorDetails {
    val requestId: String?
    val code: String?
    val message: String?
}

// Models "ErrorResponse" type in https://awslabs.github.io/smithy/1.0/spec/aws/aws-restxml-protocol.html#operation-error-serialization
internal data class XmlErrorResponse(
    val error: XmlError?,
    override val requestId: String? = error?.requestId,
) : RestXmlErrorDetails {
    override val code: String? = error?.code
    override val message: String? = error?.message
}

// Models "Error" type in https://awslabs.github.io/smithy/1.0/spec/aws/aws-restxml-protocol.html#operation-error-serialization
internal data class XmlError(
    override val requestId: String?,
    override val code: String?,
    override val message: String?
) : RestXmlErrorDetails

/**
 * Deserializes rest XML protocol errors as specified by:
 * https://awslabs.github.io/smithy/1.0/spec/aws/aws-restxml-protocol.html#error-response-serialization
 *
 * Returns parsed data in normalized form or throws IllegalArgumentException if response cannot be parsed.
 * NOTE: we use an explicit XML deserializer here because we rely on validating the root element name
 * for dealing with the alternate error response forms
 */
@InternalSdkApi
public suspend fun parseRestXmlErrorResponse(payload: ByteArray): ErrorDetails {
    val details = ErrorResponseDeserializer.deserialize(XmlDeserializer(payload, true))
        ?: XmlErrorDeserializer.deserialize(XmlDeserializer(payload, true))
        ?: throw DeserializationException("Unable to deserialize RestXml error.")
    return ErrorDetails(details.code, details.message, details.requestId)
}

/*
 * The deserializers in this file were initially generated by the SDK and then
 * adapted to fit this use case of deserializing well-known error structures from
 * restXml-based services.
 */

/**
 * Deserializes rest Xml protocol errors as specified by:
 * - Smithy spec: https://awslabs.github.io/smithy/1.0/spec/aws/aws-restxml-protocol.html#operation-error-serialization
 */
internal object ErrorResponseDeserializer {
    private val ERROR_DESCRIPTOR = SdkFieldDescriptor(SerialKind.Struct, XmlSerialName("Error"))
    private val REQUESTID_DESCRIPTOR = SdkFieldDescriptor(SerialKind.String, XmlSerialName("RequestId"))
    private val OBJ_DESCRIPTOR = SdkObjectDescriptor.build {
        trait(XmlSerialName("ErrorResponse"))
        field(ERROR_DESCRIPTOR)
        field(REQUESTID_DESCRIPTOR)
    }

    suspend fun deserialize(deserializer: Deserializer): XmlErrorResponse? {
        var requestId: String? = null
        var xmlError: XmlError? = null

        return try {
            deserializer.deserializeStruct(OBJ_DESCRIPTOR) {
                loop@ while (true) {
                    when (findNextFieldIndex()) {
                        ERROR_DESCRIPTOR.index -> xmlError = XmlErrorDeserializer.deserialize(deserializer)
                        REQUESTID_DESCRIPTOR.index -> requestId = deserializeString()
                        null -> break@loop
                        else -> skipValue()
                    }
                }
            }

            XmlErrorResponse(xmlError, requestId ?: xmlError?.requestId)
        } catch (e: DeserializationException) {
            null // return so an appropriate exception type can be instantiated above here.
        }
    }
}

/**
 * This deserializer is used for both the nested Error node from ErrorResponse as well as the top-level
 * Error node as described in https://awslabs.github.io/smithy/1.0/spec/aws/aws-restxml-protocol.html#operation-error-serialization
 */
internal object XmlErrorDeserializer {
    private val MESSAGE_DESCRIPTOR = SdkFieldDescriptor(SerialKind.String, XmlSerialName("Message"))
    private val CODE_DESCRIPTOR = SdkFieldDescriptor(SerialKind.String, XmlSerialName("Code"))
    private val REQUESTID_DESCRIPTOR = SdkFieldDescriptor(SerialKind.String, XmlSerialName("RequestId"))
    private val OBJ_DESCRIPTOR = SdkObjectDescriptor.build {
        trait(XmlSerialName("Error"))
        field(MESSAGE_DESCRIPTOR)
        field(CODE_DESCRIPTOR)
        field(REQUESTID_DESCRIPTOR)
    }

    suspend fun deserialize(deserializer: Deserializer): XmlError? {
        var message: String? = null
        var code: String? = null
        var requestId: String? = null

        return try {
            deserializer.deserializeStruct(OBJ_DESCRIPTOR) {
                loop@ while (true) {
                    when (findNextFieldIndex()) {
                        MESSAGE_DESCRIPTOR.index -> message = deserializeString()
                        CODE_DESCRIPTOR.index -> code = deserializeString()
                        REQUESTID_DESCRIPTOR.index -> requestId = deserializeString()
                        null -> break@loop
                        else -> skipValue()
                    }
                }
            }

            XmlError(requestId, code, message)
        } catch (e: DeserializationException) {
            null // return so an appropriate exception type can be instantiated above here.
        }
    }
}
