package ai.systema.model

import ai.systema.exception.EmptyStringException
import ai.systema.helper.toAttribute
import ai.systema.model.internal.Raw
import kotlinx.serialization.KSerializer
import kotlinx.serialization.Serializable
import kotlinx.serialization.builtins.serializer
import kotlinx.serialization.descriptors.SerialDescriptor
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder

/**
 * An attribute is a key in the json definition of a record.
 * Example:
 *
 * ```
 * {
 *  "color": "red"
 * }
 * ```
 *
 * This record has an attribute "color", and its value is "red".
 */
@Serializable(Attribute.Companion::class)
public data class Attribute(override val raw: String) : Raw<String> {

    init {
        if (raw.isBlank()) throw EmptyStringException("Attribute")
    }

    override fun toString(): String {
        return raw
    }

    public companion object : KSerializer<Attribute> {

        private val serializer = String.serializer()

        override val descriptor: SerialDescriptor = serializer.descriptor

        override fun serialize(encoder: Encoder, value: Attribute) {
            String.serializer().serialize(encoder, value.raw)
        }

        override fun deserialize(decoder: Decoder): Attribute {
            return serializer.deserialize(decoder).toAttribute()
        }
    }
}
