package ai.systema.connection.internal

import ai.systema.configuration.Compression
import ai.systema.configuration.Configuration
import ai.systema.configuration.DynamicConfigurationManager
import ai.systema.configuration.Impersonate
import ai.systema.configuration.RetryableHost
import ai.systema.connection.RequestOptions
import ai.systema.constants.SystemaConstants
import ai.systema.enums.CallType
import ai.systema.enums.EndpointType
import ai.systema.exception.UnreachableHostsException
import ai.systema.helper.logging.SystemaLoggerFactory
import ai.systema.model.ClientUser
import ai.systema.model.config.DynamicConfig
import ai.systema.model.request.RecommendationRequest
import ai.systema.model.request.RequestUser
import ai.systema.model.request.SmartSearchRequest
import ai.systema.model.request.SmartSuggestRequest
import io.ktor.client.features.ResponseException
import io.ktor.client.features.timeout
import io.ktor.client.request.HttpRequestBuilder
import io.ktor.client.request.request
import io.ktor.http.HttpMethod
import io.ktor.http.URLProtocol
import io.ktor.network.sockets.ConnectTimeoutException
import io.ktor.network.sockets.SocketTimeoutException
import io.ktor.utils.io.errors.IOException
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.datetime.Clock
import kotlinx.datetime.Instant
import kotlin.math.floor
import kotlin.time.ExperimentalTime

internal class Connector(
    private val configuration: Configuration,
) : Configuration by configuration,
    DynamicConfigurationManager {

    private val logger = SystemaLoggerFactory.logger(Connector::class.toString())

    private val hostStatusExpirationDelayMS: Long = 1000L * 60L * 5L
    private val mutex = Mutex()

    private val reqSettingsErrMsg = "should not be set already in the request. SDK needs to set it."

    private lateinit var dynamicConfig: DynamicConfig

    @OptIn(ExperimentalTime::class)
    private var lasRefreshedAt: Instant? = null

    private val configMutex = Mutex()

    lateinit var clientUser: ClientUser

    suspend fun callableHosts(endpointType: EndpointType, callType: CallType): List<RetryableHost> {
        return mutex.withLock {
            val candidates = hosts[endpointType] ?: listOf()
            candidates.expireHostsOlderThan(hostStatusExpirationDelayMS)
            val hostsCallType = candidates.filterCallType(callType)
            val hostsCallTypeAreUp = hostsCallType.filter { it.isUp }

            if (hostsCallTypeAreUp.isEmpty()) {
                hostsCallType.apply { forEach { it.reset() } }
            } else hostsCallTypeAreUp
        }
    }

    private fun httpRequestBuilder(
        httpMethod: HttpMethod,
        path: String,
        protocol: URLProtocol?,
        requestOptions: RequestOptions?,
        body: String?,
    ): HttpRequestBuilder {
        return HttpRequestBuilder().apply {
            url.path(path)
            url.protocol = protocol ?: URLProtocol.HTTPS
            method = httpMethod
            compress(body)
            setCredentials(configuration.credentials)
            setRequestOptions(requestOptions)
        }
    }

    private fun HttpRequestBuilder.compress(payload: String?) {
        if (payload != null) {
            body = when (compression) {
                Compression.None -> payload
                else -> throw UnsupportedOperationException("Compression is not supported yet")
            }
        }
    }

    suspend inline fun <reified T> request(
        httpMethod: HttpMethod,
        endpointType: EndpointType,
        callType: CallType,
        path: String,
        requestOptions: RequestOptions?,
        body: String? = null,
    ): T {
        val hosts = callableHosts(endpointType, callType)
        val errors by lazy(LazyThreadSafetyMode.NONE) { mutableListOf<Throwable>() }
        val requestBuilder = httpRequestBuilder(httpMethod, path, URLProtocol.HTTPS, requestOptions, body)

        if (hosts.isEmpty()) {
            errors.add(IllegalStateException("No hosts available"))
        }

        for (host in hosts) {
            requestBuilder.url.host = host.url.host
            requestBuilder.url.protocol = host.url.protocol
            requestBuilder.url.port = host.url.port
            try {
                setTimeout(requestBuilder, requestOptions, callType, host)
                return httpClient.request<T>(requestBuilder).apply {
                    mutex.withLock { host.reset() }
                }
            } catch (exception: Exception) {
                errors += exception
                when (exception) {
                    is SocketTimeoutException, is ConnectTimeoutException -> mutex.withLock { host.hasTimedOut() }
                    is IOException -> mutex.withLock { host.hasFailed() }
                    is ResponseException -> {
                        val value = exception.response.status.value
                        val isRetryable = floor(value / 100f) != 4f
                        if (isRetryable) mutex.withLock { host.hasFailed() } else throw exception
                    }
                    else -> throw exception
                }
            }
        }

        throw UnreachableHostsException(errors)
    }

    /**
     * Set socket read/write timeout.
     */
    private fun setTimeout(
        requestBuilder: HttpRequestBuilder,
        requestOptions: RequestOptions?,
        callType: CallType,
        host: RetryableHost,
    ) {
        requestBuilder.timeout {
            socketTimeoutMillis = requestOptions.getTimeout(callType) * (host.retryCount + 1)
        }
    }

    fun getDynamicConfig(): DynamicConfig {
        return this.dynamicConfig
    }

    override suspend fun initialize(requestOptions: RequestOptions?) {
        this.clientUser = Impersonate.clientUser(kvStore)
        this.refreshDynamicConfig()
    }


    @OptIn(ExperimentalTime::class)
    override suspend fun refreshDynamicConfig(requestOptions: RequestOptions?) {
        return configMutex.withLock {
            if (this.lasRefreshedAt == null || this.lasRefreshedAt!! < Clock.System.now()
                    .minus(SystemaConstants.MaxSessionDuration)
            ) {
                // TODO uncomment when dynamic configuration feature is finalized
//                logger.debug("Refreshing dynamic config")
//                this.dynamicConfig = this.request<ResponseDefault<DynamicConfig>>(
//                    HttpMethod.Get,
//                    EndpointType.DynamicConfig,
//                    CallType.Read,
//                    RouteDynamicConfig,
//                    requestOptions
//                ).result

                this.lasRefreshedAt = Clock.System.now()
                clientUser.refreshSession(kvStore)
                logger.debug("Refreshed dynamic config at: $lasRefreshedAt")
            }
        }
    }

    override suspend fun prepDynamicRoute(endpointType: EndpointType, actionType: String): String {
        this.refreshDynamicConfig()
        TODO("Implement using dynamic config")
    }

    override suspend fun prepDynamicPayload(actionType: String): String {
        this.refreshDynamicConfig()
        TODO("Implement using dynamic config")
    }

    internal suspend fun injectSystemaSettings(payload: RecommendationRequest?): RecommendationRequest {
        val req = payload ?: RecommendationRequest()

        if (req.environment != null)
            throw IllegalArgumentException("'environment' $reqSettingsErrMsg")

        if (req.user != null)
            throw IllegalArgumentException("'user' $reqSettingsErrMsg")

        req.environment = credentials.environment.value
        req.user = RequestUser(
            sid = clientUser.sessionId,
            uid = clientUser.userIdHash,
            fid = clientUser.fingerprint,
        )

        return req
    }

    internal suspend fun injectSystemaSettings(payload: SmartSuggestRequest?): SmartSuggestRequest {
        val req = payload ?: SmartSuggestRequest(query = "")

        if (req.environment != null)
            throw IllegalArgumentException("'environment' $reqSettingsErrMsg")

        if (req.user != null)
            throw IllegalArgumentException("'user' $reqSettingsErrMsg")

        req.environment = credentials.environment.value
        req.user = getRequestUser()

        return req
    }

    internal suspend fun injectSystemaSettings(payload: SmartSearchRequest?): SmartSearchRequest {
        val req = payload ?: SmartSearchRequest()

        if (req.environment != null)
            throw IllegalArgumentException("'environment' $reqSettingsErrMsg")

        if (req.user != null)
            throw IllegalArgumentException("'user' $reqSettingsErrMsg")

        req.environment = credentials.environment.value
        req.user = getRequestUser()

        return req
    }

    private suspend fun getRequestUser(): RequestUser {
        this.clientUser.refreshSession(kvStore)

        return RequestUser(
            sid = clientUser.sessionId,
            uid = clientUser.userIdHash,
            fid = clientUser.fingerprint,
        )
    }

}
