package ai.systema.android.listener

import ai.systema.android.Tagger
import ai.systema.android.findChildrenBySystemaTags
import ai.systema.android.getSystemaTagVal
import ai.systema.android.getTagMapping
import ai.systema.android.hasSystemaTag
import ai.systema.android.isVisible
import ai.systema.client.SystemaAI
import ai.systema.constants.SystemaTags
import ai.systema.helper.logging.SystemaLoggerFactory
import ai.systema.model.tracker.view.ItemContainer
import ai.systema.model.tracker.view.prepContainerItem
import android.view.View
import android.widget.AdapterView
import kotlinx.coroutines.runBlocking

public class SystemaEventListener(
    private val systema: SystemaAI,
    private val tagMapping: Map<String, Int>
) : View.OnClickListener,
    AdapterView.OnItemClickListener,
    View.OnLayoutChangeListener,
    View.OnAttachStateChangeListener {

    private var lastProductId: String = ""
    private val logger = SystemaLoggerFactory.logger(this.javaClass.name, systema.logLevel)

    override fun onClick(view: View?) {
        val productId = Tagger.getProductId(view, tagMapping)
        if (productId != null) {
            sendItemClickedEvent(
                productId = productId,
                recId = Tagger.getRecId(view, tagMapping) ?: "",
                url = Tagger.getProductUrl(view, tagMapping) ?: "",
                referrer = ""
            )
        }
    }

    override fun onItemClick(parent: AdapterView<*>?, view: View?, position: Int, id: Long) {
        onClick(view)
    }

    override fun onLayoutChange(
        v: View?,
        left: Int,
        top: Int,
        right: Int,
        bottom: Int,
        oldLeft: Int,
        oldTop: Int,
        oldRight: Int,
        oldBottom: Int,
    ) {
        val isVisible = v?.isVisible() as Boolean
        val resultId = Tagger.getResultId(v, tagMapping)
        val hasObserved = v.hasSystemaTag(SystemaTags.Observed, tagMapping) &&
                v.getSystemaTagVal(SystemaTags.Observed, tagMapping) == true.toString()

        logger.debug("Layout changed: $left $right, $top, $bottom, $oldLeft, $oldRight, $oldTop $oldBottom")
        logger.debug("IsVisible: $isVisible, resultId: $resultId, hasObserved: $hasObserved")

        if (isVisible && resultId != null && !hasObserved) {
            Tagger.setObserved(v, systema.getTagMapping(), true)
            sendContainerShownEvent(v, resultId)
        }
    }

    override fun onViewAttachedToWindow(v: View?) {
        val productId = Tagger.getProductId(v, tagMapping)
        logger.debug("Attached product view: $productId")

        if (productId != null) {
            logger.debug("Send PageView: $productId")
            sendPageViewEvent(productId = productId, url = Tagger.getProductUrl(v, tagMapping) ?: "")
        }
    }

    override fun onViewDetachedFromWindow(v: View?) {
        val productId = Tagger.getProductId(v, tagMapping)
        logger.debug("Detached product view: $productId")
    }

    private fun sendItemClickedEvent(productId: String, url: String = "", recId: String = "", referrer: String = "") {
        if (lastProductId.isNotBlank() && lastProductId == productId) {
            logger.warn("Repeated ItemClicked Event: product_id: $productId, rec_id:$recId url:$url, referrer:$referrer. Skipping this event.")
            return
        }

        runBlocking {
            logger.debug("Send ItemClicked Event: product_id: $productId, rec_id:$recId url:$url, referrer:$referrer")
            systema.trackItemClicked(
                productId = productId,
                recId = recId,
                url = url,
                referrer = referrer
            )
        }
    }

    private fun sendPageViewEvent(productId: String, url: String = "", recId: String = "", referrer: String = "") {
        if (lastProductId.isNotBlank() && lastProductId == productId) {
            logger.warn("Repeated PageViewed Event: product_id: $productId, rec_id:$recId url:$url, referrer:$referrer. Skipping this event.")
            return
        }

        runBlocking {
            logger.debug("Send PageViewed Event: product_id: $productId, rec_id:$recId url:$url, referrer:$referrer")
            systema.trackPageViewed(
                productId = productId,
                recId = recId,
                url = url,
                referrer = referrer
            )
        }
    }

    private fun sendContainerShownEvent(v: View, resultId: String) {
        // get recIds from the children of the container
        val recItems: MutableList<Map<String, String>> = mutableListOf()

        val systemaTags: Map<String, String> = mapOf(
            SystemaTags.RecId to "Any",
        )
        val candidates: MutableList<View> = mutableListOf()
        v.findChildrenBySystemaTags(systemaTags, tagMapping, candidates)

        // find candidate recIds
        for (child in candidates) {
            val hasObserved = Tagger.getObserved(child, tagMapping) == true.toString()
            if (!hasObserved) {
                Tagger.setObserved(v, tagMapping, true)
                val recId = Tagger.getRecId(child, tagMapping)
                if (recId != null) {
                    recItems.add(
                        prepContainerItem(recId)
                    )
                }
            }
        }

        // send message
        if (recItems.isNotEmpty()) {
            logger.debug("Send ContainerShown: $resultId")
            logger.debug(recItems.toString())
            val container: ItemContainer = ItemContainer(
                resultId = resultId,
                recItems = recItems,
            )

            runBlocking {
                systema.trackContainerShown(
                    listOf(container),
                    v.getSystemaTagVal(SystemaTags.ContainerUrl, tagMapping)
                )
            }
        }
    }


}
