package ai.h2o.mojos.runtime.api.backend;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/** Access to zip-compressed fileset */
public class ZipFileReaderBackend implements ReaderBackend {
    private final ZipFile zipFile;

    private ZipFileReaderBackend(ZipFile zipFile) {
        this.zipFile = zipFile;
    }

    public static ReaderBackend open(File zippedMojoFile) throws IOException {
        return new ZipFileReaderBackend(new ZipFile(zippedMojoFile));
    }

    @Override
    public InputStream getInputStream(String resourceName) throws IOException {
        final ZipEntry entry = zipFile.getEntry(resourceName);
        if (entry == null) {
            throw new FileNotFoundException(resourceName);
        }
        return zipFile.getInputStream(entry);
    }

    @Override
    public boolean exists(String resourceName) {
        return zipFile.getEntry(resourceName) != null;
    }

    @Override
    public Collection<String> list() {
        final List<String> result = new ArrayList<>(zipFile.size());
        final Enumeration<? extends ZipEntry> entries = zipFile.entries();
        while (entries.hasMoreElements()) {
            final ZipEntry zipEntry = entries.nextElement();
            result.add(zipEntry.getName());
        }
        return result;
    }

    @Override
    public void close() throws IOException {
        zipFile.close();
    }

    @Override
    public String toString() {
        return String.format("%s[%s]", getClass().getSimpleName(), zipFile.getName());
    }
}
